function varargout = nma_poisson_GUI(varargin)
%main GUI file for poisson 2D solver

% NMA_POISSON_GUI M-file for nma_poisson_GUI.fig
%      NMA_POISSON_GUI, by itself, creates a new NMA_POISSON_GUI or raises the existing
%      singleton*.

%      H = NMA_POISSON_GUI returns the handle to a new NMA_POISSON_GUI or the handle to
%      the existing singleton*.

%      NMA_POISSON_GUI('CALLBACK',hObject,eventData,handles,...) calls the local
%      function named CALLBACK in NMA_POISSON_GUI.M with the given input arguments.

%      NMA_POISSON_GUI('Property','Value',...) creates a new NMA_POISSON_GUI or raises the
%      existing singleton*.  Starting from the left, property value pairs are
%      applied to the GUI before nma_poisson_GUI_OpeningFcn gets called.  An
%      unrecognized property name or invalid value makes property application
%      stop.  All inputs are passed to nma_poisson_GUI_OpeningFcn via varargin.

%      *See GUI Options on GUIDE's Tools menu.  Choose "GUI allows only one
%      instance to run (singleton)".

% See also: GUIDE, GUIDATA, GUIHANDLES

% Copyright: Nasser M. Abbasi
% Free to use for educational and research purposes only
% part of my matlab toolbox GUI apps, solves the 2D Elliptic PDE
%

% Edit the above text to modify the response to help nma_poisson_GUI

% Last Modified by GUIDE v2.5 13-Jan-2011 02:37:41

% Begin initialization code - DO NOT EDIT
gui_Singleton = 1;
gui_State = struct('gui_Name',       mfilename, ...
   'gui_Singleton',  gui_Singleton, ...
   'gui_OpeningFcn', @nma_poisson_GUI_OpeningFcn, ...
   'gui_OutputFcn',  @nma_poisson_GUI_OutputFcn, ...
   'gui_LayoutFcn',  [] , ...
   'gui_Callback',   []);
if nargin && ischar(varargin{1})
   gui_State.gui_Callback = str2func(varargin{1});
end

if nargout
   [varargout{1:nargout}] = gui_mainfcn(gui_State, varargin{:});
else
   gui_mainfcn(gui_State, varargin{:});
end
% End initialization code - DO NOT EDIT

%to FIX: remove all gcf from code, can cause problems with mutliple figures
%Fix the arrays used to track ||r|| and the ratio, the now grow dynamically

% --- Executes just before nma_poisson_GUI is made visible.
function nma_poisson_GUI_OpeningFcn(hObject, eventdata, handles, varargin)
% This function has no output args, see OutputFcn.
% hObject    handle to figure
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
% varargin   command line arguments to nma_poisson_GUI (see VARARGIN)

%states that this program can be in at any one moment
INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;

handles.output = hObject;

set(handles.figure1, 'UserData',[]);
set(handles.figure1,'Name','2D elliptic PDE solver   By Nasser M. Abbasi, version feb 12, 2012');


data.event = 1; %EMPTY;
data.state=INIT;

data.original_axes_position_nobar = get(handles.axes,'Position');
data.original_axes_position_bar = 0;
data.colorbar_handle=0;

data.original_axes_position_nobar_force = get(handles.force_axes_tag,'Position');
data.original_axes_position_bar_force = 0;
data.colorbar_handle_force=0;


data.step_number=0;
data.A=[];
data.A_condition=0;
data.A_eig=[];


[a,map]=imread('nma_poisson_play_black.png');
[r,c,d]=size(a);
x=ceil(r/55);
y=ceil(c/55);
g=a(1:x:end,1:y:end,:);
g(g==255)=5.5*255;
set(handles.runBtn,'CData',g)

[a,map]=imread('nma_poisson_step_black.png');
[r,c,d]=size(a);
x=ceil(r/55);
y=ceil(c/55);
g=a(1:x:end,1:y:end,:);
g(g==255)=5.5*255;
set(handles.stepBtn,'CData',g)

[a,map]=imread('nma_poisson_stop_black.png');
[r,c,d]=size(a);
x=ceil(r/55);
y=ceil(c/55);
g=a(1:x:end,1:y:end,:);
g(g==255)=5.5*255;
set(handles.pauseBtn,'CData',g)

[a,map]=imread('nma_poisson_reset.jpg');
[r,c,d]=size(a);
x=ceil(r/55);
y=ceil(c/55);
g=a(1:x:end,1:y:end,:);
g(g==255)=5.5*255;
set(handles.resetBtn,'CData',g)

[a,map]=imread('nma_poisson_grid.png');
[r,c,d]=size(a);
x=ceil(r/160);
y=ceil(c/160);
g=a(1:x:end,1:y:end,:);
g(g==255)=5.5*255;
set(handles.small_grid_area,'CData',a)

update_grid(handles);
set(handles.figure1, 'UserData',data);

update_force_plot(handles);
colormap(hsv);

guidata(hObject, handles);  %write it back

%stepBtn_Callback(hObject, eventdata, handles);

%resetBtn_Callback(hObject, eventdata, handles);


% UIWAIT makes nma_poisson_GUI wait for user response (see UIRESUME)
% uiwait(handles.figure1);

% --- Outputs from this function are returned to the command line.
function varargout = nma_poisson_GUI_OutputFcn(hObject, eventdata, handles, varargin)
% varargout  cell array for returning output args (see VARARGOUT);
% hObject    handle to figure
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

varargout{1} = handles.output;

%-----------------

%-----------------
function enable_all(handles)

set(handles.steepest_btn, 'Enable','on');
set(handles.conjugate_btn, 'Enable','on');
set(handles.multi_btn, 'Enable','on');
set(handles.conj_none_btn, 'Enable','on');
set(handles.conj_ssor_btn, 'Enable','on');
set(handles.conj_multi_btn, 'Enable','on');
set(handles.conj_incomplete_btn, 'Enable','on');
set(handles.conj_drop_tag, 'Enable','on');
set(handles.mu1_tag, 'Enable','on');
set(handles.mu2_tag, 'Enable','on');
set(handles.multi_smooth_tag, 'Enable','on');


set(handles.direct_solver_tag, 'Enable','on');
set(handles.x_length_tag,'Enable','on');
set(handles.y_length_tag,'Enable','on');

set(handles.stepBtn, 'Enable', 'on');

set(handles.jacobiBtn, 'Enable', 'on');
set(handles.gaussSeidelBtn, 'Enable', 'on');
set(handles.SORBtn, 'Enable', 'on');
if get(handles.gaussSeidelBtn, 'Value') == 1
   set(handles.gaussSeidelNaturalBtn, 'Enable', 'on');
   set(handles.gaussSeidelRedBlackBtn, 'Enable', 'on');
elseif get(handles.SORBtn, 'Value') == 1
   set(handles.SORoptimalBtn, 'Enable', 'on');
   set(handles.SORuserBtn, 'Enable', 'on');
   if get(handles.SORuserBtn, 'Value') == 1
      set(handles.SORuserInput, 'Enable', 'on');
   end
end

set(handles.tolInput, 'Enable', 'on');
set(handles.forceField, 'Enable', 'on');
set(handles.zero_u_btn, 'Enable', 'on');

set(handles.random_u_btn, 'Enable', 'on');

set(handles.hx_tag, 'Enable', 'on');
set(handles.hy_tag, 'Enable', 'on');

% type are the radio buttons
set(handles.south_bc_type_D, 'Enable', 'on');
set(handles.north_bc_type_D, 'Enable', 'on');
set(handles.east_bc_type_D, 'Enable', 'on');
set(handles.west_bc_type_D, 'Enable', 'on');

set(handles.south_bc_type_V, 'Enable', 'on');
set(handles.north_bc_type_V, 'Enable', 'on');
set(handles.east_bc_type_V, 'Enable', 'on');
set(handles.west_bc_type_V, 'Enable', 'on');

if get(handles.south_bc_type_D,'Value')==1
   set(handles.south_bc_tag,'Enable','on');
   set(handles.south_bc_v_tag,'Enable','off');
else
   set(handles.south_bc_tag,'Enable','off');
   set(handles.south_bc_v_tag,'Enable','on');
end

if get(handles.north_bc_type_D,'Value')==1
   set(handles.north_bc_tag,'Enable','on');
   set(handles.north_bc_v_tag,'Enable','off');
else
   set(handles.north_bc_tag,'Enable','off');
   set(handles.north_bc_v_tag,'Enable','on');
end


if get(handles.west_bc_type_D,'Value')==1
   set(handles.west_bc_tag,'Enable','on');
   set(handles.west_bc_v_tag,'Enable','off');
else
   set(handles.west_bc_tag,'Enable','off');
   set(handles.west_bc_v_tag,'Enable','on');
end

if get(handles.east_bc_type_D,'Value')==1
   set(handles.east_bc_tag,'Enable','on');
   set(handles.east_bc_v_tag,'Enable','off');
else
   set(handles.east_bc_tag,'Enable','off');
   set(handles.east_bc_v_tag,'Enable','on');
end

%-----------------

%-----------------
function disable_all(handles)

set(handles.steepest_btn, 'Enable','off');
set(handles.conjugate_btn, 'Enable','off');
set(handles.multi_btn, 'Enable','off');
set(handles.conj_none_btn, 'Enable','off');
set(handles.conj_ssor_btn, 'Enable','off');
set(handles.conj_multi_btn, 'Enable','off');
set(handles.conj_incomplete_btn, 'Enable','off');
set(handles.conj_drop_tag, 'Enable','off');
set(handles.mu1_tag, 'Enable','off');
set(handles.mu2_tag, 'Enable','off');
set(handles.multi_smooth_tag, 'Enable','off');


set(handles.direct_solver_tag, 'Enable','off');
set(handles.x_length_tag,'Enable','off');
set(handles.y_length_tag,'Enable','off');

set(handles.jacobiBtn, 'Enable', 'off');
set(handles.gaussSeidelBtn, 'Enable', 'off');
set(handles.SORBtn, 'Enable', 'off');
set(handles.gaussSeidelNaturalBtn, 'Enable', 'off');
set(handles.gaussSeidelRedBlackBtn, 'Enable', 'off');
set(handles.SORoptimalBtn, 'Enable', 'off');
set(handles.SORuserBtn, 'Enable', 'off');
set(handles.SORuserInput, 'Enable', 'off');

set(handles.tolInput, 'Enable', 'off');
set(handles.forceField, 'Enable', 'off');
set(handles.zero_u_btn, 'Enable', 'off');

set(handles.random_u_btn, 'Enable', 'off');

set(handles.hx_tag, 'Enable', 'off');
set(handles.hy_tag, 'Enable', 'off');

set(handles.south_bc_tag, 'Enable', 'off');
set(handles.north_bc_tag, 'Enable', 'off');
set(handles.east_bc_tag, 'Enable', 'off');
set(handles.west_bc_tag, 'Enable', 'off');

set(handles.south_bc_v_tag, 'Enable', 'off');
set(handles.north_bc_v_tag, 'Enable', 'off');
set(handles.east_bc_v_tag, 'Enable', 'off');
set(handles.west_bc_v_tag, 'Enable', 'off');


set(handles.south_bc_type_D, 'Enable', 'off');
set(handles.north_bc_type_D, 'Enable', 'off');
set(handles.east_bc_type_D, 'Enable', 'off');
set(handles.west_bc_type_D, 'Enable', 'off');

set(handles.south_bc_type_V, 'Enable', 'off');
set(handles.north_bc_type_V, 'Enable', 'off');
set(handles.east_bc_type_V, 'Enable', 'off');
set(handles.west_bc_type_V, 'Enable', 'off');

%-------------------------------
function [AZ,EL,status]=get_view(handles)

status=1;

AZ=get(handles.AZ_slider_tag,'Value');
EL=get(handles.EL_slider_tag,'Value');
set(handles.AZ_tag,'String',sprintf('%3.3f',AZ));
set(handles.EL_tag,'String',sprintf('%3.3f',EL));

%----------------------------------------

function v=adjust_colorbar(colorbar_handle,original_axes_position_bar,...
   original_axes_position_nobar,handles,ax)

v=colorbar_handle;

if get(handles.colorbar_tag,'Value')==0
   if colorbar_handle ~=0
      if ishandle(colorbar_handle)
         colorbar(colorbar_handle,'delete');
         v=0;
         set(ax,'Position',original_axes_position_nobar);
      else
         v=0;
      end
   end
else
   if original_axes_position_bar~=0
      set(ax,'Position',original_axes_position_bar);
   end
end


%-----------------------------
function [data,status,hh]=update_plots_2(data,handles,d,the_title,ax,is_force)
status=1;
hh=0;

set(handles.figure1, 'CurrentAxes',ax);

contents = cellstr(get(handles.plot_type_popup_tag,'String'));
plot_type = contents{get(handles.plot_type_popup_tag,'Value')};
shading_name = get_current_shading(handles);

if not(is_force)
   data.colorbar_handle=adjust_colorbar...
      (data.colorbar_handle,data.original_axes_position_bar,...
      data.original_axes_position_nobar,handles,ax);
else
   data.colorbar_handle_force=adjust_colorbar(data.colorbar_handle_force,...
      data.original_axes_position_bar_force,...
      data.original_axes_position_nobar_force,handles,ax);
end

if strcmpi(plot_type,'mesh')||strcmpi(plot_type,'meshc')...
      || strcmpi(plot_type,'surfc')||strcmpi(plot_type,'surfl')||...
      strcmpi(plot_type,'surf')||strcmpi(plot_type,'plot3')
   [AZ,EL,status]=get_view(handles);
   if not(status)
      return
   end
   
   if not(is_force)
      if data.step_number==1 || get(handles.auto_z_tag,'Value')==1
         hh=feval(str2func(plot_type),ax, data.X, data.Y,d);
         r=zlim(ax);
         data.last_max_z_value=r(2);
         data.last_min_z_value=r(1);
      else
         if get(handles.fixed_z_tag,'Value')==1
            hh=feval(str2func(plot_type),ax, data.X, data.Y, d);
            zlim(ax,[data.last_min_z_value data.last_max_z_value]);
         end
      end
   else
      hh=feval(str2func(plot_type),ax, data.X, data.Y, d);
   end
   
   if get(handles.hidden_on_tag,'Value')==0
      hidden off;
   else
      hidden on;
   end
   
   if get(handles.show_grid_tag,'Value')==1
      hold on;
      plot3(ax,data.X,data.Y,d,'.');
      hold off;
   end
   
   shading(shading_name);
   view(ax,[AZ,EL]);
   if get(handles.colorbar_tag,'Value')==1
      if is_force
         data.colorbar_handle_force=colorbar('EastOutside','FontSize',7);
         data.original_axes_position_bar_force=get(ax,'Position');
      else
         data.colorbar_handle=colorbar('EastOutside','FontSize',7);
         data.original_axes_position_bar=get(handles.axes,'Position');
      end
   end
else
   [n,~]= get_normalized_contour_level(handles.contour_level_tag);
   [C,h] = contour(ax,data.X, data.Y, d,n);
   if get(handles.add_label_tag,'Value')==1
      clabel(C,h,'FontSize',7,'Color','k','Rotation',0,'LabelSpacing',72);
   end
   
   if get(handles.show_grid_tag,'Value')==1
      hold(ax);
      plot3(ax,data.X,data.Y,zeros(size(data.X)),'.');
      hold(ax);
   end
   
end
title(ax, the_title);
xlabel(ax, 'x (south side)');
ylabel(ax, 'y (west side)');


%------------------------------
function  data = update_plots(data, handles)

contents = cellstr(get(handles.which_plot_tag,'String')); % returns which_plot_tag contents as cell array
which_plot=contents{get(handles.which_plot_tag,'Value')};% returns selected item from which_plot_tag

contents = cellstr(get(handles.plot_type_popup_tag,'String'));
plot_type = contents{get(handles.plot_type_popup_tag,'Value')};

%states
INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;

if get(handles.fixed_z_tag,'Value')==1
   if data.step_number==1
      data.last_max_z_value=max(max(data.u));
      data.last_min_z_value=min(min(data.u));
   end
end

set(0,'CurrentFigure',handles.figure1);
grid on;

switch which_plot
   case 'solution'
      [data,status,~] = update_plots_2(data,handles,data.u,'u(x,y)',handles.axes,false);
      if not(status)
         return
      end
   case 'residual'
      [data,status,~] = update_plots_2(data,handles,data.res,'u(x,y)',handles.axes,false);
      if not(status)
         return
      end
end

if strcmpi(plot_type,'mesh')||strcmpi(plot_type,'meshc')...
      || strcmpi(plot_type,'surfc')||strcmpi(plot_type,'surfl')||...
      strcmpi(plot_type,'surf')||strcmpi(plot_type,'plot3')
   [data,status,hh] = update_plots_2(data,handles,data.f,'f(x,y)',handles.force_axes_tag,true);
   if not(status)
      return
   end
else
   if not(is_constant_function(get(handles.forceField, 'String')))
      [~,hh]=contour(handles.force_axes_tag, data.X, data.Y, data.f);
      set(hh,'HitTest','off');
      if get(handles.show_grid_tag,'Value')==1
         hold on;
         plot3(handles.force_axes_tag,data.X,data.Y,zeros(size(data.X)),'.');
         hold off;
      end
   else
      hh=mesh(handles.force_axes_tag, data.X, data.Y, data.f);
      set(hh,'HitTest','off');
      if get(handles.show_grid_tag,'Value')==1
         hold on;
         plot3(handles.force_axes_tag,data.X,data.Y,data.f,'.');
         hold off;
      end
   end
end

colormap(get_colormap_color_as_string(handles));
drawnow();

%--------------------------
function L2 = lap2d(nx,ny,hx,hy)
Lx=lap1dy(nx,hx,hy);  %makes the MAIN block, only one block
Ly=lap1dx(ny,hx,hy);  %makes the off block, only one block

Ix=speye(nx);
Iy=speye(ny);

Lm=kron(Iy,Lx); %does the central diagonal
Lo=kron(Ly,Ix); %does the off diagonal
L2=Lm+Lo;

%------------------------
function L=lap1dy(n,hx,hy)
e=ones(n,1);
T1=2*(hx^2+hy^2);
B=[-e*hy^2 (1/2)*T1*e -e*hy^2];
L=spdiags(B,[-1 0 1],n,n);

%-------------------------------------
function L=lap1dx(n,hx,hy)
e=ones(n,1);
T1=2*(hx^2+hy^2);
B=[-e*hx^2 (1/2)*T1*e -e*hx^2];
L=spdiags(B,[-1 0 1],n,n);

%----------------------------------
function [A,b]=make_system_for_direct_solver(data)

ny=data.ny;
hy=data.hy;
hx=data.hx;

if data.west_bc_type == 1 || data.east_bc_type==1 || ...
      data.north_bc_type==1 || data.south_bc_type==1
   
   nzmax = nzmax_for_sparse(data);
   return;
   
   %    nx=data.nx+1;
   %    N = (nx-2)*(ny-2);
   %    A = lap2d_west_bc_type_1(nx-2,ny-2,hx,hy);
   
else
   nx=data.nx;
   N= (nx-2)*(ny-2);
   A= lap2d(nx-2,ny-2,hx,hy);
   
   u=data.u;
   
   T1=2*(hx^2+hy^2);
   T2=(hx^2*hy^2);
   b=zeros(N,1);
   
   f=data.f;
   
   for i=2:ny-1
      ii=i-1;
      for j=2:nx-1
         jj=j-1;
         k=(ii-1)*(nx-2)+jj;
         
         if i==2
            if j==2
               b(k)=-T2*f(i,j)+hy^2*u(i,j-1)+hx^2*u(i-1,j);
            else
               if j==nx-1
                  b(k)=-T2*f(i,j)+hy^2*u(i,j+1)+hx^2*u(i-1,j);
               else
                  b(k)=-T2*f(i,j)+hx^2*u(i-1,j);
               end
            end
         else
            if i==ny-1
               if j==2
                  b(k)=-T2*f(i,j)+hy^2*u(i,j-1)+hx^2*u(i+1,j);
               else
                  if j==nx-1
                     b(k)=-T2*f(i,j)+hy^2*u(i,j+1)+hx^2*u(i+1,j);
                  else
                     b(k)=-T2*f(i,j)+hx^2*u(i+1,j);
                  end
               end
            else
               if j==2
                  b(k)=-T2*f(i,j)+hy^2*u(i,j-1);
               else
                  if j==nx-1
                     b(k)=-T2*f(i,j)+hy^2*u(i,j+1);
                  else
                     b(k)=-T2*f(i,j);
                  end
               end
            end
         end
      end
   end
   
end

%------------------------------------------------
function [data, status] = direct_solver(data, handles)

INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;

status = 1;

[data, status] = common_init_step(data, handles);
if not(status)
   return
end

if data.west_bc_type == 1 || data.east_bc_type==1 || ...
      data.north_bc_type ==1 || data.south_bc_type==1
   
   data.step_number=0;
   data= post_step_common(data, handles);
   data.state=CONVERGED;
   
   errordlg('Sorry, in this version, direct solver only supports Dirichlet boundary conditions',...
      'Bad Input', 'modal');
   uicontrol(handles.west_bc_type_D);
   return
end

set(handles.omega_tag, 'String', 'N/A');
drawnow();

data.u = set_boundary_conditions(data,data.u);

[data.A,data.b] = make_system_for_direct_solver(data);
data.A_condition=condest(data.A);
set(handles.A_condition_tag,'String',sprintf('%5.2f',data.A_condition));
%data.A_eig=eigs(data.A);  %too slow

%this below due to change of indexing. I will redo everything later to
%use consistant idexing so I do not have to do these things. But the
%problem is that the textbook used natural indexing, which is not
%programming friendly, and for direct solver I did not use that, so
%some parts in the code uses natural and direct solver does not.
sol=data.A\data.b;
for i=2:data.ny-1
   ii=i-1;
   from = (ii-1)*(data.nx-2)+1;
   to=from+(data.nx-2)-1;
   data.u(i,2:end-1)=sol(from:to);
end

data.step_number=0;
data= post_step_common(data, handles);
data.state=CONVERGED;


%------------------------------
function [data, status] = make_step(data, handles)

INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;


if get(handles.jacobiBtn, 'Value') == 1
   [data, status] = make_jacobi_step(data, handles);
elseif get(handles.gaussSeidelBtn, 'Value') == 1
   [data, status] = make_GS_step(data, handles);
elseif get(handles.SORBtn, 'Value') == 1
   [data, status] = make_SOR_step(data, handles);
elseif get(handles.direct_solver_tag,'Value')==1
   [data, status] = direct_solver(data, handles);
elseif get(handles.steepest_btn,'Value')==1
   [data, status] = steepest_descent(data, handles);
elseif get(handles.conjugate_btn,'Value')==1
   data.step_number=0;
   data.state=CONVERGED;
   resetBtn_Callback(0, 0, handles)   
   
   errordlg('Sorry, this option not currently available',...
      'Bad Input', 'modal');
   uicontrol(handles.conjugate_btn);
   return
   %   [data, status] = conjugate_gradient(data, handles);
elseif get(handles.multi_btn,'Value')==1
   data.step_number=0;
   data.state=CONVERGED;
   resetBtn_Callback(0, 0, handles)      
   errordlg('Sorry, this option not currently available',...
      'Bad Input', 'modal');
   uicontrol(handles.conjugate_btn);
   return
   %[data, status] = multi_grid(data, handles);
end

%------------------------------------------
function [n, status] = verify_valid_AZ_or_EL(str, h, msg)

status = 1;
[n, status] = str2num(str);

if not(status)
   errordlg(sprintf('non - numeric value for %s',msg),...
      'Bad Input', 'modal');
   uicontrol(h);
   return
end

if abs(n) > 180
   errordlg(sprintf('%s value must be less than 180 %s',msg),...
      'Bad Input', 'modal');
   uicontrol(h);
   return
end

%------------------------
function clear_all_plots(handles)
cla(handles.axes, 'reset');
cla(handles.axesRelativeR, 'reset');
cla(handles.resid_ratio_axes, 'reset');

set(handles.resid_ratio_tag,'String','');
set(handles.resid_tag,'String','');
set(handles.iterTag,'String','');
set(handles.omega_tag,'String','');

update_grid(handles);

%-------------------------
function data = initialize_boundary_conditions(data,handles)
switch get(get(handles.west_bc_type, 'SelectedObject'), 'Tag') % Get Tag of selected object.
   
   case 'west_bc_type_D'
      data.west_bc_type = 0; %direchilet
      
      [data.west_bc_value,status]= ...
         evaluate_2D_function(get(handles.west_bc_tag, 'String'),data.X, data.Y);
      
      if not(status)
         errordlg('west side Dirichlet function not valid\nDid you forget the dot? as in .* instead of just * ?',...
            'Bad Input', 'modal');
         uicontrol(handles.west_bc_tag);
         return
      end
      
      data.west_bc_value = data.west_bc_value(:,1);
      data.u(:,1) = data.west_bc_value;
      
   case 'west_bc_type_V'
      data.west_bc_type = 1;
      
      [data.west_bc_value,status]= ...
         evaluate_2D_function(get(handles.west_bc_v_tag, 'String'),data.X, data.Y);
      
      if not(status)
         errordlg('west side Neumann  function not valid\nDid you forget the dot? as in .* instead of just * ?',...
            'Bad Input', 'modal');
         uicontrol(handles.west_bc_v_tag);
         return
      end
      
      data.west_bc_value = data.west_bc_value(:,1);
      
end

switch get(get(handles.east_bc_type, 'SelectedObject'), 'Tag') % Get Tag of selected object.
   case 'east_bc_type_D'
      data.east_bc_type = 0; %direchilet
      
      [data.east_bc_value,status]= ...
         evaluate_2D_function(get(handles.east_bc_tag, 'String'),data.X, data.Y);
      
      if not(status)
         errordlg('east side Dirichlet function not valid\nDid you forget the dot? as in .* instead of just * ?',...
            'Bad Input', 'modal');
         uicontrol(handles.east_bc_tag);
         return
      end
      
      data.east_bc_value = data.east_bc_value(:,end);
      data.u(:,end) = data.east_bc_value;
      
   case 'east_bc_type_V'
      data.east_bc_type = 1;
      
      [data.east_bc_value,status]= ...
         evaluate_2D_function(get(handles.east_bc_v_tag, 'String'),data.X, data.Y);
      
      if not(status)
         errordlg('east side Neumann  function not valid\nDid you forget the dot? as in .* instead of just * ?',...
            'Bad Input', 'modal');
         uicontrol(handles.east_bc_v_tag);
         return
      end
      
      data.east_bc_value = data.east_bc_value(:,end);
end

switch get(get(handles.north_bc_type, 'SelectedObject'), 'Tag') % Get Tag of selected object.
   case 'north_bc_type_D'
      data.north_bc_type = 0; %direchilet
      
      [data.north_bc_value,status]= ...
         evaluate_2D_function(get(handles.north_bc_tag, 'String'),data.X, data.Y);
      
      if not(status)
         errordlg('north side Dirichlet function not valid\nDid you forget the dot? as in .* instead of just * ?',...
            'Bad Input', 'modal');
         uicontrol(handles.north_bc_tag);
         return
      end
      
      data.north_bc_value = data.north_bc_value(end,:);
      
      if data.west_bc_type==0
         data.u(end,1)=(data.u(end,1)+data.north_bc_value(1))/2;
      else
         data.u(end,1)=data.north_bc_value(1);
      end
      
      if data.east_bc_type==0
         data.u(end,end)=(data.u(end,end)+data.north_bc_value(end))/2;
      else
         data.u(1,end)=data.north_bc_value(end);
      end
      
      data.u(end,2:end-1) = data.north_bc_value(2:end-1);
      
   case 'north_bc_type_V'
      data.north_bc_type = 1;
      
      [data.north_bc_value,status]= ...
         evaluate_2D_function(get(handles.north_bc_v_tag, 'String'),data.X, data.Y);
      
      if not(status)
         errordlg('north side Neumann  function not valid\nDid you forget the dot? as in .* instead of just * ?',...
            'Bad Input', 'modal');
         uicontrol(handles.north_bc_v_tag);
         return
      end
      
      data.north_bc_value = data.north_bc_value(end,:);
      
end

switch get(get(handles.south_bc_type, 'SelectedObject'), 'Tag') % Get Tag of selected object.
   case 'south_bc_type_D'
      data.south_bc_type = 0; %direchilet
      
      [data.south_bc_value,status]= ...
         evaluate_2D_function(get(handles.south_bc_tag, 'String'),data.X, data.Y);
      
      if not(status)
         errordlg('south side Dirichlet function not valid\nDid you forget the dot? as in .* instead of just * ?',...
            'Bad Input', 'modal');
         uicontrol(handles.south_bc_tag);
         return
      end
      
      data.south_bc_value = data.south_bc_value(1,:);
      
      if data.west_bc_type==0
         data.u(1,1)=(data.u(1,1)+data.south_bc_value(1))/2;
      else
         data.u(1,1)=data.south_bc_value(1);
      end
      
      if data.east_bc_type==0
         data.u(1,end)=(data.u(1,end)+data.south_bc_value(end))/2;
      else
         data.u(1,end)=data.south_bc_value(end);
      end
      
      data.u(1,2:end-1) = data.south_bc_value(2:end-1);
      
   case 'south_bc_type_V'
      data.south_bc_type = 1;
      
      [data.south_bc_value,status]= ...
         evaluate_2D_function(get(handles.south_bc_v_tag, 'String'),data.X, data.Y);
      
      if not(status)
         errordlg('south side Neumann  function not valid\nDid you forget the dot? as in .* instead of just * ?',...
            'Bad Input', 'modal');
         uicontrol(handles.south_bc_v_tag);
         return
      end
      
      data.south_bc_value = data.south_bc_value(1,:);
end

%--------------------------
function [data, status] = common_init_step(data, handles)

clear_all_plots(handles);

tolStr = get(handles.tolInput, 'String');
[data.tol, status] = nma_verify_valid_positive_numeric(tolStr, handles.tolInput, 'tolerance');
if not(status)
   return
end

[v,~] = get_normalized_mesh_spacing(handles.hx_tag);
data.hx = v;
[len,~]=get_normalized_length(handles.x_length_tag);
intervals_x = len/v;
data.nx = intervals_x + 1;

[v,~] = get_normalized_mesh_spacing(handles.hy_tag);
data.hy = v;
[len,~]=get_normalized_length(handles.y_length_tag);
intervals_y = len/v;
data.ny = intervals_y + 1;


[data.X, data.Y] = get_meshgrid(0,data.hx,data.hx * intervals_x,...
   0,data.hy,data.hy * intervals_y);

force_as_string = get(handles.forceField, 'String');
[data.f,status] = evaluate_2D_function(force_as_string,data.X, data.Y);

if not(status)
   data.f = 0;
   errordlg('force function is not valid Matlab synatx \nDid you forget the dot? as in .* instead of just * ?',...
      'Bad Input', 'modal');
   uicontrol(handles.forceField);
   return
end

data.normf = norm( data.f ); % find norm
data.res   = zeros(data.ny, data.nx);     % storage for residual

switch get(get(handles.initial_u_tag, 'SelectedObject'), 'Tag') % Get Tag of selected object.
   case 'zero_u_btn'
      data.u = data.res;
   case 'random_u_btn'
      data.u = rand(data.ny, data.nx);
end

data.current_resid_norm = 0;
data.last_resid_norm = 0;
data.resid_ratio = 0;

data.cy = data.hy^2;
data.cx = data.hx^2;
data.c1 = 1/(2*(data.cx + data.cy));
data.cxy= data.cx * data.cy;

data = initialize_boundary_conditions(data,handles);

data.ratio_data = [];
data.resid_norm_data = [];  %FIX ME, not auto expand

if is_all_neumann(data)
   status = 0;
   errordlg('All boundary condition can not be Neumann, at least one must be dirichlet',...
      'Bad Input', 'modal');
   uicontrol(handles.west_bc_tag);
else
   status = 1;
end

%--------------------------
function status = is_all_neumann(data)

if data.south_bc_type == 1 && ...
      data.north_bc_type == 1 && ...
      data.east_bc_type == 1 && ...
      data.west_bc_type == 1
   status = true;
else
   status = false;
end

%-----------------------------
function data = post_step_steepest(data, handles)
%states
INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;

data.step_number = data.step_number + 1;
set(handles.iterTag, 'String', data.step_number);

the_norm=norm(data.steepest_r,2);
set(handles.resid_tag, 'String', the_norm);

data.resid_norm_data(data.step_number)= data.rel;
plot(handles.axesRelativeR,1:data.step_number,data.resid_norm_data);
xlabel(handles.axesRelativeR, 'iteration');
refreshdata(handles.axesRelativeR, 'caller');
drawnow;


if the_norm < data.tol
   data.state = CONVERGED;
   data.step_number=0;
   set(handles.state_tag, 'String', 'converged !');
end

%-------------------------
function data= post_step_common(data, handles)

%states
INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;

data.step_number = data.step_number + 1;
set(handles.iterTag, 'String', data.step_number);

data.resid_norm_data(data.step_number)= data.rel;
plot(handles.axesRelativeR,1:data.step_number,data.resid_norm_data);
xlabel(handles.axesRelativeR, 'iteration');
refreshdata(handles.axesRelativeR, 'caller');
drawnow;

if data.step_number ==1
   data.last_resid_norm = data.rel;
   data.resid_ratio = 0;
elseif data.step_number ==2
   data.current_resid_norm = data.rel;
   data.resid_ratio = data.rel/data.last_resid_norm;
else
   data.last_resid_norm = data.current_resid_norm;
   data.current_resid_norm = data.rel;
   data.resid_ratio = data.current_resid_norm/data.last_resid_norm;
end

set(handles.resid_tag, 'String', data.rel);
set(handles.resid_ratio_tag, 'String', data.resid_ratio);

if data.step_number > 1
   data.ratio_data(data.step_number-1) =  data.resid_ratio;
   plot(handles.resid_ratio_axes,2:data.step_number,data.ratio_data);
   xlabel(handles.resid_ratio_axes, 'iteration');
   refreshdata(handles.resid_ratio_axes, 'caller');
   drawnow;
end

if data.rel < data.tol
   data.state = CONVERGED;
   data.step_number=0;
   set(handles.state_tag, 'String', 'converged !');
end

%-----------------------------
function [data, status]= make_SOR_step(data, handles)

status = 1;

if data.step_number == 0 %initial
   [data, status] = common_init_step(data, handles);
   if not(status)
      return
   end
   
   data.unew = data.u;
   
   data.isOptimalOmega = get(handles.SORoptimalBtn, 'Value');
   if data.isOptimalOmega
      t = cos(pi * data.hx) + cos(pi * data.hy);
      syms x;
      r = double(solve( t^2 * x^2 - 16 * x + 16, x));
      data.omega = min(r);
      %data.omega=2*(1- pi*data.h);
   else
      data.omega = get(handles.SORuserInput, 'String');
      [data.omega, status] = str2num( data.omega);
      
      if not(status)
         errordlg('You must enter a numeric value for SOR omega',...
            'Bad Input', 'modal');
         uicontrol(handles.SORuserInput);
         return
      end
      
      if abs(data.omega) < 2 * eps
         errordlg('You must enter non - zero value for SOR omega',...
            'Bad Input', 'modal');
         uicontrol(handles.SORuserInput);
         return
      end
      
      if data.omega < 0
         errordlg('SOR omega must be positive',...
            'Bad Input', 'modal');
         uicontrol(handles.SORuserInput);
         return
      end
      
      
      if data.omega > 2
         errordlg('SOR omega cant be more than 2',...
            'Bad Input', 'modal');
         uicontrol(handles.SORuserInput);
         return
      end
   end
   
end

data.u = set_boundary_conditions(data,data.u);
data.unew = set_boundary_conditions(data,data.unew);
tmp=(data.hx^2+data.hy^2)/(data.hx^2*data.hy^2);

for i = 2:(data.ny - 1)
   for j = 2:(data.nx - 1)
      
     data.res(i,j) = data.f(i,j) -...
         ( data.unew(i,j-1)/data.hx^2 + ...
         data.u(i,j+1)/data.hx^2 - ...
         2*tmp*data.u(i,j)+...
         data.unew(i-1,j)/data.hy^2+...         
         data.u(i+1,j)/data.hy^2);
     
     data.unew(i, j)=data.u(i,j)-data.omega*(1/(2*tmp))*data.res(i,j);
     
      %term = ( data.unew(i,j-1)+data.u(i,j+1))*data.hy^2 +...
      %   (data.unew(i-1,j)+data.u(i+1,j))*data.hx^2 - ...
      %   (data.hx^2*data.hy^2)*data.f(i, j);
      
      %term = data.omega/(2*(data.hx^2+data.hy^2)) * term;
      %data.unew(i, j)=term + (1 - data.omega)*data.u(i, j);
      
      grid_point_update(data,handles,i,j);
      
   end
end

data.u = data.unew;

%for i=2:(data.ny-1)
%   for j=2:(data.nx-1)
%      
%      data.res(i,j) = data.f(i,j) -( data.u(i,j-1)-2*data.u(i,j)+data.u(i,j+1))/data.hx^2 - ...
%         (data.u(i-1,j)-2*data.u(i,j)+data.u(i+1,j))/data.hy^2;
%      
%   end
%end


if abs(data.normf) < eps
   data.rel = norm(data.res);
else
   data.rel = norm(data.res)/data.normf;
end

data = post_step_common(data, handles);
set(handles.omega_tag, 'String', sprintf('%f',data.omega));

%--------------------------
function [data, status] = make_GS_step(data, handles)

status = 1;

if data.step_number == 0 %initial
   [data, status] = common_init_step(data, handles);
   if not(status)
      return
   end
   
   data.isRedBlack = get(handles.gaussSeidelRedBlackBtn, 'Value');
   data.unew = data.u;
end

data.u = set_boundary_conditions(data,data.u);
data.unew = set_boundary_conditions(data,data.unew);

T1=1/(2*(data.cx + data.cy));

if data.isRedBlack
   for i = 2:(data.ny - 1)
      for j = 2:(data.nx - 1)
         if bitget(i + j, 1) == 0  % red squares
            data.u(i, j)   = T1* (  (data.u(i,j-1)+data.u(i,j+1))*data.cy +...
               (data.u(i-1,j)+data.u(i+1,j))*data.cx - ...
               data.cx * data.cy * data.f(i, j) );
            
            grid_point_update(data,handles,i,j);
         end
      end
   end
   
   for i = 2:(data.ny - 1)
      for j = 2:(data.nx - 1)
         
         if bitget(i + j, 1) ~=0  % black squares
            data.u(i, j)   = T1* (  (data.u(i,j-1)+data.u(i,j+1))*data.cy +...
               (data.u(i-1,j)+data.u(i+1,j))*data.cx - ...
               data.cx * data.cy * data.f(i, j) );
            
            grid_point_update(data,handles,i,j);
         end
         
      end
   end
else %natural ordering
   for i = 2:(data.ny - 1)
      for j = 2:(data.nx - 1)
         
         data.unew(i, j)   = T1* (  (data.unew(i,j-1)+data.u(i,j+1))*data.cy +...
            (data.unew(i-1,j)+data.u(i+1,j))*data.cx - ...
            data.cx * data.cy * data.f(i, j) );
         
         grid_point_update(data,handles,i,j);
      end
   end
   data.u = data.unew;
end


for i=2:(data.ny-1)
   for j=2:(data.nx-1)
      
      data.res(i,j) = data.f(i,j) -( data.u(i,j-1)-2*data.u(i,j)+data.u(i,j+1))/data.hx^2 - ...
         (data.u(i-1,j)-2*data.u(i,j)+data.u(i+1,j))/data.hy^2;
      
   end
end

if abs(data.normf) < eps
   data.rel = norm(data.res);
else
   data.rel = norm(data.res)/data.normf;
end


data = post_step_common(data, handles);
set(handles.omega_tag, 'String', 'N/A');

%--------------------------
function u = set_boundary_conditions(data,u)

T1=1/(2*(data.hx^2+data.hy^2));
T2=(data.hx^2*data.hy^2)*T1;
T3=(data.hx*data.hy)/(data.hx^2+data.hy^2);

if data.west_bc_type==1  %neumann
   j=1;
   if max(abs(data.west_bc_value(2:end-1)))<2*eps
      % homogenouse numann
      for i=2:data.ny-1
         u(i,j) =T3*( data.hx/(2*data.hy)*...
            (u(i+1,j)+u(i-1,j))+...
            data.hy/data.hx * u(i,j+1)) - ...
            T2*data.f(i,j);
      end
   else %non-homogenous numann
      for i=2:data.ny-1
         u(i,j) =T1*( 2*u(i,j+1)*data.hy^2+...
            (  u(i-1,j)+ u(i+1,j) )*data.hx^2+...
            2*data.hy^2 * data.hx * data.west_bc_value(i) )...
            -T2*data.f(i,j);
      end
   end
end

if data.east_bc_type==1
   j=data.nx;
   if max(abs(data.east_bc_value(2:end-1)))<2*eps
      % homogenouse numann
      for i=2:data.ny-1
         u(i,j) =T3*( data.hx/(2*data.hy)*...
            (u(i+1,j)+u(i-1,j))+...
            data.hy/data.hx *u(i,j-1)) - ...
            T2*data.f(i,j);
      end
   else
      for i=2:data.ny-1
         u(i,j) =T1*( 2*u(i,j-1)*data.hy^2+...
            (  u(i-1,j)+ data.u(i+1,j) )*data.hx^2+...
            2*data.hy^2 * data.hx * data.east_bc_value(i) )...
            -T2*data.f(i,j);
      end
   end
end

if data.north_bc_type==1
   i=data.ny;
   if max(abs(data.north_bc_value(2:end-1)))<2*eps
      % homogenouse numann
      for j=2:data.nx-1
         u(i,j) =T3*( data.hy/(2*data.hx)*...
            (u(i,j-1)+u(i,j+1))+...
            data.hx/data.hy * u(i-1,j)) - ...
            T2*data.f(i,j);
      end
   else
      for j=2:data.nx-1
         u(i,j) =T1*( 2*u(i-1,j)*data.hx^2+...
            (  u(i,j-1)+ u(i,j+1) )*data.hy^2+...
            2*data.hx^2 * data.hy * data.north_bc_value(j) )...
            -T2*data.f(i,j);
      end
   end
   
   if data.west_bc_type==1 %correction         FIX ME, I am using
      %the same equation for corner with zero numman, but this can be
      %numman which is not zero. check
      u(data.ny,1)= (2*data.hx*data.hy)/(data.hx^2+data.hy^2)*...
         (data.hx/(2*data.hy)*u(end-1,1) + data.hy/(2*data.hx)*u(end,2));
   end
   
   if data.east_bc_type==1
      u(end,end)= (2*data.hx*data.hy)/(data.hx^2+data.hy^2)*...
         (data.hx/(2*data.hy)*u(end-1,end) + data.hy/(2*data.hx)*u(end,end-1));
   end
end


if data.south_bc_type==1
   i=1;
   if max(abs(data.south_bc_value(2:end-1)))<2*eps
      % homogenouse numann
      for j=2:data.nx-1
         u(i,j) =T3*( data.hy/(2*data.hx)*...
            (u(i,j-1)+u(i,j+1))+...
            data.hx/data.hy *u(i+1,j)) - ...
            T2*data.f(i,j);
      end
   else
      for j=2:data.nx-1
         u(i,j) =T1*( 2*u(i+1,j)*data.hx^2+...
            (  u(i,j-1)+ u(i,j+1) )*data.hy^2+...
            2*data.hx^2 * data.hy * data.south_bc_value(j) )...
            -T2*data.f(i,j);
      end
   end
   
   if data.west_bc_type==1
      u(1,1)= (2*data.hx*data.hy)/(data.hx^2+data.hy^2)*...
         (data.hx/(2*data.hy)*u(2,1) + data.hy/(2*data.hx)*u(1,2));
   end
   
   if data.east_bc_type==1
      u(1,end)= (2*data.hx*data.hy)/(data.hx^2+data.hy^2)*...
         (data.hx/(2*data.hy)*u(2,end) + data.hy/(2*data.hx)*u(1,end-1));
   end
   
end

%--------------------------
function [data, status] = make_jacobi_step(data, handles)

status = 1;

if data.step_number == 0 %initial
   [data, status] = common_init_step(data, handles);
   if not(status)
      return
   end
end

data.u = set_boundary_conditions(data,data.u);

T1 = 1/(2*(data.hx^2 + data.hy^2));

% is this right below?? I am using the updated value while looping??

for i=2:(data.ny-1)
   for j=2:(data.nx-1)
      data.u(i,j) = T1 *( (data.u(i,j-1)+data.u(i,j+1))*data.hy^2 + ...
         (data.u(i-1,j)+data.u(i+1,j))*data.hx^2 - ...
         data.hx^2*data.hy^2*data.f(i, j) );
      
      grid_point_update(data,handles,i,j);
   end
end

for i=2:(data.ny-1)
   for j=2:(data.nx-1)
      data.res(i,j) = data.f(i,j) -( data.u(i,j-1)-2*data.u(i,j)+data.u(i,j+1))/data.hx^2 - ...
         (data.u(i-1,j)-2*data.u(i,j)+data.u(i+1,j))/data.hy^2;
   end
end

if abs(data.normf) < eps
   data.rel = norm(data.res);
else
   data.rel = norm(data.res)/data.normf;
end


data = post_step_common(data, handles);
set(handles.omega_tag, 'String', 'N/A');

%---------------------------
function [data, status] = steepest_descent(data, handles)

INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;

status = 1;

if data.step_number == 0 %initial
   [data, status] = common_init_step(data, handles);
   if not(status)
      return
   end
   
   set(handles.omega_tag, 'String', 'N/A');
   drawnow();
   
   if data.west_bc_type == 1 || data.east_bc_type==1 || ...
         data.north_bc_type ==1 || data.south_bc_type==1
      
      data.step_number=0;
      data= post_step_common(data, handles);
      data.state=CONVERGED;
      
      errordlg('Sorry, in this version, steepest descent only supports Dirichlet boundary conditions',...
         'Bad Input', 'modal');
      uicontrol(handles.west_bc_type_D);
      return
   end
   
   data.u = set_boundary_conditions(data,data.u);
   [data.A,data.b] = make_system_for_direct_solver(data);
   data.A_condition=condest(data.A);
   set(handles.A_condition_tag,'String',sprintf('%5.2f',data.A_condition));
   
   if not(nma_ISSPD(data.A))
      errordlg('steepest descent, something is wrong, A is not SPD !',...
         'Bad Input', 'modal');
      uicontrol(handles.steepest_btn);
   end
   
   [nRowF,nColF] = size(data.b(:));
   
   if nColF ~= 1
      errordlg('steepest descent, something is wrong, f must be a vector !',...
         'Bad Input', 'modal');
      uicontrol(handles.steepest_btn);
   end
   
   [n,~] = size(data.A);
   if n ~= nRowF
      errordlg('steepest descent, something is wrong, f size not compatible with A size !',...
         'Bad Input', 'modal');
      uicontrol(handles.steepest_btn);
   end
   
   data.steepest_sol=zeros(n,1);
   data.steepest_r = data.b;
end

w     = data.A*data.steepest_r;
alpha = (data.steepest_r'*data.steepest_r)/(data.steepest_r'*w);
data.steepest_sol = data.steepest_sol + alpha*data.steepest_r;
data.steepest_r   = data.steepest_r - alpha*w;

for i=2:data.ny-1
   ii=i-1;
   from = (ii-1)*(data.nx-2)+1;
   to=from+(data.nx-2)-1;
   data.u(i,2:end-1)=data.steepest_sol(from:to);
end

data.rel = norm(data.steepest_r,2);

data= post_step_common(data, handles);

%data = post_step_steepest(data, handles);


%--------------------------------------
function  grid_point_update(data,handles,i,j)

if get(handles.grid_point_tag,'Value')==1
   %update_plots(data,handles);
   
   set(handles.y_coord_tag,'String',...
      sprintf('%3.3f',(i-1)*data.hy));
   
   set(handles.x_coord_tag,'String',...
      sprintf('%3.3f',(j-1)*data.hx));
   
   drawnow();
   
   %refreshdata(handles.x_coord_tag);
   %refreshdata(handles.y_coord_tag);
   
end



% --- Executes on button press in jacobiBtn.
function jacobiBtn_Callback(hObject, eventdata, handles)
% hObject    handle to jacobiBtn (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of jacobiBtn
handles.output = hObject;
set(handles.jacobiBtn, 'Value', 1);
set(handles.direct_solver_tag, 'Value', 0);
set(handles.gaussSeidelBtn, 'Value', 0);
set(handles.SORBtn, 'Value', 0);

set(handles.gaussSeidelNaturalBtn, 'Enable', 'off');
set(handles.gaussSeidelNaturalBtn, 'Value', 1);

set(handles.gaussSeidelRedBlackBtn, 'Enable', 'off');
set(handles.SORoptimalBtn, 'Enable', 'off');
set(handles.SORuserBtn, 'Enable', 'off');

set(handles.conjugate_btn,'Value',0);
set(handles.steepest_btn,'Value',0);
set(handles.multi_btn,'Value',0);

% Update handles structure
guidata(hObject, handles);


% --- Executes on button press in gaussSeidelBtn.
function gaussSeidelBtn_Callback(hObject, eventdata, handles)
% hObject    handle to gaussSeidelBtn (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of gaussSeidelBtn

handles.output = hObject;
set(handles.jacobiBtn, 'Value', 0);
set(handles.direct_solver_tag, 'Value', 0);
set(handles.gaussSeidelBtn, 'Value', 1);
set(handles.SORBtn, 'Value', 0);

set(handles.gaussSeidelNaturalBtn, 'Enable', 'on');
set(handles.gaussSeidelRedBlackBtn, 'Enable', 'on');
set(handles.SORoptimalBtn, 'Enable', 'off');
set(handles.SORuserBtn, 'Enable', 'off');

set(handles.conjugate_btn,'Value',0);
set(handles.steepest_btn,'Value',0);
set(handles.multi_btn,'Value',0);

% Update handles structure
guidata(hObject, handles);

function SORuserInput_Callback(hObject, eventdata, handles)
% hObject    handle to SORuserInput (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of SORuserInput as text
%        str2double(get(hObject,'String')) returns contents of SORuserInput as a double


% --- Executes during object creation, after setting all properties.
function SORuserInput_CreateFcn(hObject, eventdata, handles)
% hObject    handle to SORuserInput (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject, 'BackgroundColor'), get(0, 'defaultUicontrolBackgroundColor'))
   set(hObject, 'BackgroundColor', 'white');
end



function forceField_Callback(hObject, eventdata, handles)
% hObject    handle to forceField (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of forceField as text
%        str2double(get(hObject,'String')) returns contents of forceField as a double


% --- Executes during object creation, after setting all properties.
function forceField_CreateFcn(hObject, eventdata, handles)
% hObject    handle to forceField (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject, 'BackgroundColor'), get(0, 'defaultUicontrolBackgroundColor'))
   set(hObject, 'BackgroundColor', 'white');
end


% --- Executes on button press in SORBtn.
function SORBtn_Callback(hObject, eventdata, handles)
% hObject    handle to SORBtn (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of SORBtn
handles.output = hObject;
set(handles.jacobiBtn, 'Value', 0);
set(handles.direct_solver_tag, 'Value', 0);
set(handles.gaussSeidelBtn, 'Value', 0);
set(handles.SORBtn, 'Value', 1);

set(handles.gaussSeidelNaturalBtn, 'Enable', 'off');
set(handles.gaussSeidelRedBlackBtn, 'Enable', 'off');
set(handles.SORoptimalBtn, 'Enable', 'on');
set(handles.SORuserBtn, 'Enable', 'on');
if get(handles.SORuserBtn, 'Value') == 1
   set(handles.SORuserInput, 'Enable', 'on');
end

set(handles.conjugate_btn,'Value',0);
set(handles.steepest_btn,'Value',0);
set(handles.multi_btn,'Value',0);


% Update handles structure
guidata(hObject, handles);


% --- Executes on button press in gaussSeidelNaturalBtn.
function gaussSeidelNaturalBtn_Callback(hObject, eventdata, handles)
% hObject    handle to gaussSeidelNaturalBtn (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

handles.output = hObject;

set(handles.gaussSeidelNaturalBtn, 'Value', 1);
set(handles.gaussSeidelRedBlackBtn, 'Value', 0);

guidata(hObject, handles);

% Hint: get(hObject,'Value') returns toggle state of gaussSeidelNaturalBtn


% --- Executes on button press in gaussSeidelRedBlackBtn.
function gaussSeidelRedBlackBtn_Callback(hObject, eventdata, handles)
% hObject    handle to gaussSeidelRedBlackBtn (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

handles.output = hObject;

set(handles.gaussSeidelNaturalBtn, 'Value', 0);
set(handles.gaussSeidelRedBlackBtn, 'Value', 1);

guidata(hObject, handles);

% Hint: get(hObject,'Value') returns toggle state of gaussSeidelRedBlackBtn

% --- Executes on button press in SORoptimalBtn.
function SORoptimalBtn_Callback(hObject, eventdata, handles)
% hObject    handle to SORoptimalBtn (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

handles.output = hObject;

set(handles.SORoptimalBtn, 'Value', 1);
set(handles.SORuserBtn, 'Value', 0);
set(handles.SORuserInput, 'Enable', 'off');
% Update handles structure
guidata(hObject, handles);

% Hint: get(hObject,'Value') returns toggle state of SORoptimalBtn

% --- Executes on button press in SORuserBtn.
function SORuserBtn_Callback(hObject, eventdata, handles)
% hObject    handle to SORuserBtn (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

handles.output = hObject;

set(handles.SORoptimalBtn, 'Value', 0);
set(handles.SORuserBtn, 'Value', 1);
set(handles.SORuserInput, 'Enable', 'on');
% Update handles structure
guidata(hObject, handles);

function tolInput_Callback(hObject, eventdata, handles)
% hObject    handle to tolInput (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of tolInput as text
%        str2double(get(hObject,'String')) returns contents of tolInput as a double


% --- Executes during object creation, after setting all properties.
function tolInput_CreateFcn(hObject, eventdata, handles)
% hObject    handle to tolInput (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject, 'BackgroundColor'), get(0, 'defaultUicontrolBackgroundColor'))
   set(hObject, 'BackgroundColor', 'white');
end


% --- Executes on button press in pauseBtn.
function pauseBtn_Callback(hObject, eventdata, handles)
% hObject    handle to pauseBtn (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

%states
INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;

handles.output = hObject;
data=get(handles.figure1,'UserData');
if data.state == RUNNING
   set(handles.stepBtn,'Enable','on');
end

data.state = PAUSED;
set(handles.figure1,'UserData',data);



function iterTag_Callback(hObject, eventdata, handles)
% hObject    handle to iterTag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of iterTag as text
%        str2double(get(hObject,'String')) returns contents of iterTag as a double


% --- Executes during object creation, after setting all properties.
function iterTag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to iterTag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject, 'BackgroundColor'), get(0, 'defaultUicontrolBackgroundColor'))
   set(hObject, 'BackgroundColor', 'white');
end




function resid_tag_Callback(hObject, eventdata, handles)
% hObject    handle to resid_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of resid_tag as text
%        str2double(get(hObject,'String')) returns contents of resid_tag as a double


% --- Executes during object creation, after setting all properties.
function resid_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to resid_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject, 'BackgroundColor'), get(0, 'defaultUicontrolBackgroundColor'))
   set(hObject, 'BackgroundColor', 'white');
end


function resid_ratio_tag_Callback(hObject, eventdata, handles)
% hObject    handle to resid_ratio_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of resid_ratio_tag as text
%        str2double(get(hObject,'String')) returns contents of resid_ratio_tag as a double


% --- Executes during object creation, after setting all properties.
function resid_ratio_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to resid_ratio_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject, 'BackgroundColor'), get(0, 'defaultUicontrolBackgroundColor'))
   set(hObject, 'BackgroundColor', 'white');
end



function omega_tag_Callback(hObject, eventdata, handles)
% hObject    handle to omega_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of omega_tag as text
%        str2double(get(hObject,'String')) returns contents of omega_tag as a double


% --- Executes during object creation, after setting all properties.
function omega_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to omega_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject, 'BackgroundColor'), get(0, 'defaultUicontrolBackgroundColor'))
   set(hObject, 'BackgroundColor', 'white');
end

function state_tag_Callback(hObject, eventdata, handles)
% hObject    handle to state_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of state_tag as text
%        str2double(get(hObject,'String')) returns contents of state_tag as a double


% --- Executes during object creation, after setting all properties.
function state_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to state_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject, 'BackgroundColor'), get(0, 'defaultUicontrolBackgroundColor'))
   set(hObject, 'BackgroundColor', 'white');
end


function hx_tag_Callback(hObject, eventdata, handles)
% hObject    handle to hx_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of hx_tag as text
%        str2double(get(hObject,'String')) returns contents of hx_tag as a double

[v, I] = get_normalized_mesh_spacing(handles.hx_tag);

set(handles.hx_tag, 'Value', I - 1);
set(handles.hx_value_tag, 'String', sprintf('%0.5f',v));
%set(handles.hx_value_bin_tag,'String',sprintf('2^
set(handles.hx_value_bin_tag, 'String', sprintf('%d',1/v+1));


update_grid(handles);


% --- Executes during object creation, after setting all properties.
function hx_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to hx_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject, 'BackgroundColor'), get(0, 'defaultUicontrolBackgroundColor'))
   set(hObject, 'BackgroundColor', 'white');
end





function hy_tag_Callback(hObject, eventdata, handles)
% hObject    handle to hy_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of hy_tag as text
%        str2double(get(hObject,'String')) returns contents of hy_tag as a double

[v, I] = get_normalized_mesh_spacing(handles.hy_tag);

set(handles.hy_tag, 'Value', I - 1);
set(handles.hy_value_tag, 'String', sprintf('%0.5f',v));
set(handles.hy_value_bin_tag, 'String', sprintf('%d',1/v+1));

update_grid(handles);


%-------------------------

%---------------------------
function update_grid(handles)

hx = get_normalized_mesh_spacing(handles.hx_tag);
hy = get_normalized_mesh_spacing(handles.hy_tag);

[X, Y] = get_meshgrid(0,hx,1,0,hy,1);

Z = ones(size(X));
mesh(handles.axes, X, Y, Z, 'FaceColor',[.6 .6 .6]);
view(handles.axes,[0 90]);
axis(handles.axes, 'off');
refreshdata(handles.axes, 'caller');
drawnow;


%------------------------------

%-----------------------------
function  [v, I] = get_normalized_mesh_spacing(h)
A = [2^-7 2^-6 2^-5 2^-4 2^-3 2^-2 2^-1];
x = get(h, 'Value');
x = 2^-(7 - x);
B = abs(x - A);
[~,I] = min(B);
v = A(I);


% --- Executes during object creation, after setting all properties.
function hy_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to hy_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject, 'BackgroundColor'), get(0, 'defaultUicontrolBackgroundColor'))
   set(hObject, 'BackgroundColor', 'white');
end

% --- Executes on slider movement.
function speedTag_Callback(hObject, eventdata, handles)
% hObject    handle to speedTag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'Value') returns position of slider
%        get(hObject,'Min') and get(hObject,'Max') to determine range of slider

set(handles.speed_value_tag, 'String', sprintf('%3.1f',get(handles.speedTag,'Value')));


% --- Executes during object creation, after setting all properties.
function speedTag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to speedTag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: slider controls usually have a light gray background.
if isequal(get(hObject, 'BackgroundColor'), get(0, 'defaultUicontrolBackgroundColor'))
   set(hObject, 'BackgroundColor',[.9 .9 .9]);
end



function speed_value_tag_Callback(hObject, eventdata, handles)
% hObject    handle to speed_value_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of speed_value_tag as text
%        str2double(get(hObject,'String')) returns contents of speed_value_tag as a double


% --- Executes during object creation, after setting all properties.
function speed_value_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to speed_value_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject, 'BackgroundColor'), get(0, 'defaultUicontrolBackgroundColor'))
   set(hObject, 'BackgroundColor', 'white');
end


function hx_value_tag_Callback(hObject, eventdata, handles)
% hObject    handle to hx_value_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of hx_value_tag as text
%        str2double(get(hObject,'String')) returns contents of hx_value_tag as a double


% --- Executes during object creation, after setting all properties.
function hx_value_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to hx_value_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject, 'BackgroundColor'), get(0, 'defaultUicontrolBackgroundColor'))
   set(hObject, 'BackgroundColor', 'white');
end



function hy_value_tag_Callback(hObject, eventdata, handles)
% hObject    handle to hy_value_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of hy_value_tag as text
%        str2double(get(hObject,'String')) returns contents of hy_value_tag as a double


% --- Executes during object creation, after setting all properties.
function hy_value_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to hy_value_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject, 'BackgroundColor'), get(0, 'defaultUicontrolBackgroundColor'))
   set(hObject, 'BackgroundColor', 'white');
end



function hx_value_bin_tag_Callback(hObject, eventdata, handles)
% hObject    handle to hx_value_bin_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of hx_value_bin_tag as text
%        str2double(get(hObject,'String')) returns contents of hx_value_bin_tag as a double


% --- Executes during object creation, after setting all properties.
function hx_value_bin_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to hx_value_bin_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject, 'BackgroundColor'), get(0, 'defaultUicontrolBackgroundColor'))
   set(hObject, 'BackgroundColor', 'white');
end



function hy_value_bin_tag_Callback(hObject, eventdata, handles)
% hObject    handle to hy_value_bin_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of hy_value_bin_tag as text
%        str2double(get(hObject,'String')) returns contents of hy_value_bin_tag as a double


% --- Executes during object creation, after setting all properties.
function hy_value_bin_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to hy_value_bin_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject, 'BackgroundColor'), get(0, 'defaultUicontrolBackgroundColor'))
   set(hObject, 'BackgroundColor', 'white');
end



function north_bc_tag_Callback(hObject, eventdata, handles)
% hObject    handle to north_bc_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of north_bc_tag as text
%        str2double(get(hObject,'String')) returns contents of north_bc_tag as a double


% --- Executes during object creation, after setting all properties.
function north_bc_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to north_bc_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject, 'BackgroundColor'), get(0, 'defaultUicontrolBackgroundColor'))
   set(hObject, 'BackgroundColor', 'white');
end



function south_bc_tag_Callback(hObject, eventdata, handles)
% hObject    handle to south_bc_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of south_bc_tag as text
%        str2double(get(hObject,'String')) returns contents of south_bc_tag as a double


% --- Executes during object creation, after setting all properties.
function south_bc_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to south_bc_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject, 'BackgroundColor'), get(0, 'defaultUicontrolBackgroundColor'))
   set(hObject, 'BackgroundColor', 'white');
end



function east_bc_tag_Callback(hObject, eventdata, handles)
% hObject    handle to east_bc_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of east_bc_tag as text
%        str2double(get(hObject,'String')) returns contents of east_bc_tag as a double


% --- Executes during object creation, after setting all properties.
function east_bc_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to east_bc_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject, 'BackgroundColor'), get(0, 'defaultUicontrolBackgroundColor'))
   set(hObject, 'BackgroundColor', 'white');
end



function west_bc_tag_Callback(hObject, eventdata, handles)
% hObject    handle to west_bc_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of west_bc_tag as text
%        str2double(get(hObject,'String')) returns contents of west_bc_tag as a double


% --- Executes during object creation, after setting all properties.
function west_bc_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to west_bc_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject, 'BackgroundColor'), get(0, 'defaultUicontrolBackgroundColor'))
   set(hObject, 'BackgroundColor', 'white');
end


% --- Executes on selection change in plot_type_popup_tag.
function plot_type_popup_tag_Callback(hObject, eventdata, handles)
% hObject    handle to plot_type_popup_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: contents = cellstr(get(hObject,'String')) returns plot_type_popup_tag contents as cell array
%        contents{get(hObject,'Value')} returns selected item from plot_type_popup_tag

%states
INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;
data=get(handles.figure1,'UserData');

if data.state ~= INIT
   update_plots(data, handles);
else
   update_force_plot(handles);
end


% --- Executes during object creation, after setting all properties.
function plot_type_popup_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to plot_type_popup_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: popupmenu controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end


% --- Executes on mouse press over axes background.
function force_axes_tag_ButtonDownFcn(hObject, eventdata, handles)
% hObject    handle to force_axes_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

update_force_plot(handles);


%---------------------
%
%------------------------
function update_force_plot(handles)

data=get(handles.figure1,'UserData');

[v,~] = get_normalized_mesh_spacing(handles.hx_tag);
data.hx = v;
[len,~]=get_normalized_length(handles.x_length_tag);
intervals_x = len/v;
data.nx = intervals_x + 1;

[v,~] = get_normalized_mesh_spacing(handles.hy_tag);
data.hy = v;
[len,~]=get_normalized_length(handles.y_length_tag);
intervals_y = len/v;
data.ny = intervals_y + 1;

[data.X, data.Y] = get_meshgrid(0,data.hx,data.hx * intervals_x,...
   0,data.hy,data.hy * intervals_y);

h = handles.plot_type_popup_tag;
set(handles.force_axes_tag,'NextPlot','replacechildren');

force_as_string=get(handles.forceField, 'String');
[f,status] = evaluate_2D_function(force_as_string,data.X,data.Y);

if not(status)
   errordlg('force function is not valid Matlab synatx. Did you forget the dot? as in .* instead of just *?',...
      'Bad Input', 'modal');
   set(h,'HitTest','off');
   uicontrol(h);
   return
end

contents = cellstr(get(h,'String'));
v = contents{get(h,'Value')};

set(handles.figure1,'UserData',data);
data=get(handles.figure1,'UserData');
[data,~,hh]= update_plots_2(data,handles,f,'f(x,y)',handles.force_axes_tag,true);

% if strcmpi(v,'contour')
%       if not(is_constant_function(get(handles.forceField, 'String')))
%          [~,h]=contour(handles.force_axes_tag, X, Y, f);
%       else
%          h=mesh(handles.force_axes_tag, X, Y, f);
%       end
% else
%       h=feval(str2func(v),handles.force_axes_tag, X, Y,f);
% end

set(hh,'HitTest','off');
%
% data.colorbar_handle=adjust_colorbar(data.colorbar_handle,...
%    data.original_axes_position_bar_force,...
%    data.original_axes_position_nobar_force,...
%    handles,...
%    handles.force_axes_tag);
%
drawnow;
set(handles.figure1,'UserData',data);


% --- Executes on key press with focus on runBtn and none of its controls.
function runBtn_KeyPressFcn(hObject, eventdata, handles)
% hObject    handle to runBtn (see GCBO)
% eventdata  structure with the following fields (see UICONTROL)
%	Key: name of the key that was pressed, in lower case
%	Character: character interpretation of the key(s) that was pressed
%	Modifier: name(s) of the modifier key(s) (i.e., control, shift) pressed
% handles    structure with handles and user data (see GUIDATA)



function east_bc_v_tag_Callback(hObject, eventdata, handles)
% hObject    handle to east_bc_v_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of east_bc_v_tag as text
%        str2double(get(hObject,'String')) returns contents of east_bc_v_tag as a double


% --- Executes during object creation, after setting all properties.
function east_bc_v_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to east_bc_v_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end



function north_bc_v_tag_Callback(hObject, eventdata, handles)
% hObject    handle to north_bc_v_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of north_bc_v_tag as text
%        str2double(get(hObject,'String')) returns contents of north_bc_v_tag as a double


% --- Executes during object creation, after setting all properties.
function north_bc_v_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to north_bc_v_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end



function south_bc_v_tag_Callback(hObject, eventdata, handles)
% hObject    handle to south_bc_v_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of south_bc_v_tag as text
%        str2double(get(hObject,'String')) returns contents of south_bc_v_tag as a double


% --- Executes during object creation, after setting all properties.
function south_bc_v_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to south_bc_v_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end



function west_bc_v_tag_Callback(hObject, eventdata, handles)
% hObject    handle to west_bc_v_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of west_bc_v_tag as text
%        str2double(get(hObject,'String')) returns contents of west_bc_v_tag as a double



% --- Executes during object creation, after setting all properties.
function west_bc_v_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to west_bc_v_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end


% --- Executes when selected object is changed in west_bc_type.
function west_bc_type_SelectionChangeFcn(hObject, eventdata, handles)
% hObject    handle to the selected object in west_bc_type
% eventdata  structure with the following fields (see UIBUTTONGROUP)
%	EventName: string 'SelectionChanged' (read only)
%	OldValue: handle of the previously selected object or empty if none was selected
%	NewValue: handle of the currently selected object
% handles    structure with handles and user data (see GUIDATA)

if get(handles.west_bc_type_D,'Value')==1
   set(handles.west_bc_tag,'Enable','on');
   set(handles.west_bc_v_tag,'Enable','off');
else
   set(handles.west_bc_tag,'Enable','off');
   set(handles.west_bc_v_tag,'Enable','on');
end


% --- Executes when selected object is changed in east_bc_type.
function east_bc_type_SelectionChangeFcn(hObject, eventdata, handles)
% hObject    handle to the selected object in east_bc_type
% eventdata  structure with the following fields (see UIBUTTONGROUP)
%	EventName: string 'SelectionChanged' (read only)
%	OldValue: handle of the previously selected object or empty if none was selected
%	NewValue: handle of the currently selected object
% handles    structure with handles and user data (see GUIDATA)

if get(handles.east_bc_type_D,'Value')==1
   set(handles.east_bc_tag,'Enable','on');
   set(handles.east_bc_v_tag,'Enable','off');
else
   set(handles.east_bc_tag,'Enable','off');
   set(handles.east_bc_v_tag,'Enable','on');
end


% --- Executes when selected object is changed in north_bc_type.
function north_bc_type_SelectionChangeFcn(hObject, eventdata, handles)
% hObject    handle to the selected object in north_bc_type
% eventdata  structure with the following fields (see UIBUTTONGROUP)
%	EventName: string 'SelectionChanged' (read only)
%	OldValue: handle of the previously selected object or empty if none was selected
%	NewValue: handle of the currently selected object
% handles    structure with handles and user data (see GUIDATA)

if get(handles.north_bc_type_D,'Value')==1
   set(handles.north_bc_tag,'Enable','on');
   set(handles.north_bc_v_tag,'Enable','off');
else
   set(handles.north_bc_tag,'Enable','off');
   set(handles.north_bc_v_tag,'Enable','on');
end


% --- Executes when selected object is changed in south_bc_type.
function south_bc_type_SelectionChangeFcn(hObject, eventdata, handles)
% hObject    handle to the selected object in south_bc_type
% eventdata  structure with the following fields (see UIBUTTONGROUP)
%	EventName: string 'SelectionChanged' (read only)
%	OldValue: handle of the previously selected object or empty if none was selected
%	NewValue: handle of the currently selected object
% handles    structure with handles and user data (see GUIDATA)

if get(handles.south_bc_type_D,'Value')==1
   set(handles.south_bc_tag,'Enable','on');
   set(handles.south_bc_v_tag,'Enable','off');
else
   set(handles.south_bc_tag,'Enable','off');
   set(handles.south_bc_v_tag,'Enable','on');
end

%-------------------
%
%----------------------
function [v,status] = evaluate_2D_function(str,X,Y)

force_string_input= strtrim(str);

if strcmp(force_string_input, '0') || strcmp(force_string_input, '')
   force_string = '0 .* X';
else
   force_string =force_string_input;
end

force_string = ['@(X, Y)' force_string];

status = 1;
v=0;

try
   forceHandle = str2func(force_string);
   v=forceHandle(X, Y);
   [n,~]=size(v);
   if n==1  % it is a consnt
      v = v * ones(size(X));  %both X and Y have same size always
   end
catch ME
   status = 0;
   return;
end


% --- Executes on selection change in which_plot_tag.
function which_plot_tag_Callback(hObject, eventdata, handles)
% hObject    handle to which_plot_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

INIT = 1; RUNNING = 2; PAUSED = 3;
data=get(handles.figure1,'UserData');
if data.state ~=INIT
   [~]=update_plots(  data, handles);
end


% --- Executes during object creation, after setting all properties.
function which_plot_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to which_plot_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: popupmenu controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end



function AZ_tag_Callback(hObject, eventdata, handles)
% hObject    handle to AZ_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of AZ_tag as text
%        str2double(get(hObject,'String')) returns contents of AZ_tag as a double


% --- Executes during object creation, after setting all properties.
function AZ_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to AZ_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end

function EL_tag_Callback(hObject, eventdata, handles)
% hObject    handle to EL_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of EL_tag as text
%        str2double(get(hObject,'String')) returns contents of EL_tag as a double


% --- Executes during object creation, after setting all properties.
function EL_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to EL_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end



% --- Executes on button press in defaut_view_tag.
function defaut_view_tag_Callback(hObject, eventdata, handles)
% hObject    handle to defaut_view_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDAT

set(handles.AZ_tag,'String',sprintf('%3.2f',-37.5));
set(handles.EL_tag,'String',sprintf('%3.2f',30));
set(handles.AZ_slider_tag,'Value',-37.5);
set(handles.EL_slider_tag,'Value',30);
%states
INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;

data = get(handles.figure1, 'UserData');

if (data.state ~= INIT)
   [~]=update_plots(data, handles);
end

% --- Executes on slider movement.
function AZ_slider_tag_Callback(hObject, eventdata, handles)
% hObject    handle to AZ_slider_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'Value') returns position of slider
%        get(hObject,'Min') and get(hObject,'Max') to determine range of slider

INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;

AZ=get(hObject,'Value');
set(handles.AZ_tag,'String',sprintf('%3.3f',AZ));

data = get(handles.figure1, 'UserData');

if (data.state ~= INIT)
   [~]=update_plots(data, handles);
end


% --- Executes during object creation, after setting all properties.
function AZ_slider_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to AZ_slider_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: slider controls usually have a light gray background.
if isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor',[.9 .9 .9]);
end



% --- Executes on slider movement.
function EL_slider_tag_Callback(hObject, eventdata, handles)
% hObject    handle to EL_slider_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'Value') returns position of slider
%        get(hObject,'Min') and get(hObject,'Max') to determine range of slider

INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;

EL=get(hObject,'Value');
set(handles.EL_tag,'String',sprintf('%3.3f',EL));

data = get(handles.figure1, 'UserData');

if (data.state ~= INIT)
   [~]=update_plots(data, handles);
end


% --- Executes during object creation, after setting all properties.
function EL_slider_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to EL_slider_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: slider controls usually have a light gray background.
if isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor',[.9 .9 .9]);
end

%---------------------------------------
function [X,Y]= get_meshgrid(from_x,hx,to_x,from_y,hy,to_y)

[X,Y]=meshgrid(from_x:hx:to_x,from_y:hy:to_y);


% --- Executes on button press in small_grid_area.
function small_grid_area_Callback(hObject, eventdata, handles)
% hObject    handle to small_grid_area (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)


% --- If Enable == 'on', executes on mouse press in 5 pixel border.
% --- Otherwise, executes on mouse press in 5 pixel border or over stepBtn.
function stepBtn_ButtonDownFcn(hObject, eventdata, handles)
% hObject    handle to stepBtn (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

%---------------------------------------
function status = is_constant_function(str)

r1=strfind(str,'X');
r2=strfind(str,'Y');
if isempty(r1)&&isempty(r2)
   status=1;
else
   status=0;
end



% --- Executes on button press in stepBtn.
function stepBtn_Callback(hObject, eventdata, handles)
% hObject    handle to stepBtn (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

%states
INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;

set(handles.runBtn,'Enable','off');
handles.output = hObject;
data = get(handles.figure1, 'UserData');
disable_all(handles);

if data.state == CONVERGED
   data.state = INIT;
else
   data.state = RUNNING;
end

set(handles.state_tag, 'String', 'starting step...');
[data, status] = make_step(data, handles);
set(handles.state_tag, 'String', 'finished step...');
if status
   data=update_plots(data, handles);
   if data.state == CONVERGED
      set(handles.state_tag, 'String', 'converged !');
   else
      set(handles.state_tag, 'String', 'ready...');
      data.state=PAUSED;
   end
else
   data.state = INIT;
   data.step_number=0;
   enable_all(handles);
end

set(handles.figure1, 'UserData',data);
set(handles.runBtn,'Enable','on');


% --- Executes on button press in runBtn.
function runBtn_Callback(hObject, eventdata, handles)
% hObject    handle to runBtn (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

%states
INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;

set(handles.stepBtn,'Enable','off');
handles.output = hObject;
data = get(handles.figure1, 'UserData');

if data.state==CONVERGED
   data.state=INIT;
end

if data.state == INIT || data.state==PAUSED
   data.state = RUNNING;
   set(handles.figure1,'UserData',data);
   disable_all(handles);
end

while (data.state ~=PAUSED && data.state ~= CONVERGED)
   set(handles.state_tag, 'String', 'starting step...');
   [data, status] = make_step(data, handles);
   set(handles.state_tag, 'String', 'finsihed step...');
   if not(status)
      data.state = INIT;
      data.step_number=0;
      enable_all(handles);
      break;
   else
      data=update_plots(data,handles);
      tmp = get(handles.figure1, 'UserData');
      if tmp.state == PAUSED || tmp.state== INIT
         data.state = tmp.state;
         
         if data.state == INIT
            clear_all_plots(handles);
            update_grid(handles);
            data.step_number=0;
            set(handles.state_tag, 'String', 'ready...');
            enable_all(handles);
         else
            set(handles.stepBtn,'Enable','on');
            set(handles.state_tag, 'String', 'paused...');
         end
         
         set(handles.figure1,'UserData',data);
         break;
      else
         set(handles.state_tag, 'String', 'running...');
         set(handles.figure1,'UserData',data);
         if data.state ~= CONVERGED
            pause_amount = get(handles.speedTag, 'Value');
            if pause_amount >0
               pause(pause_amount);
            end
         end
      end
   end
end

if data.state == CONVERGED
   set(handles.state_tag, 'String', 'converged !');
   assignin('base','u',data.u);
   assignin('base','X',data.X);
   assignin('base','Y',data.Y);
end

set(handles.colorbar_tag,'Value',0);

data.colorbar_handle=adjust_colorbar...
   (data.colorbar_handle,data.original_axes_position_bar,...
   data.original_axes_position_nobar,handles,handles.axes);

data.colorbar_handle_force=adjust_colorbar(data.colorbar_handle_force,...
   data.original_axes_position_bar_force,...
   data.original_axes_position_nobar_force,handles,handles.force_axes_tag);


set(handles.stepBtn,'Enable','on');
set(handles.figure1, 'UserData',data);


% --- Executes on button press in resetBtn.
function resetBtn_Callback(hObject, eventdata, handles)
% hObject    handle to resetBtn (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

%states
INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;

handles.output = hObject;
data=get(handles.figure1,'UserData');
old_state = data.state;
data.state = INIT;
data.step_number=0;
set(handles.figure1,'UserData',data);

if old_state ~= RUNNING
   update_grid(handles);
   clear_all_plots(handles);
   enable_all(handles);
   set(handles.state_tag, 'String', 'ready...');
end

data=get(handles.figure1,'UserData');
set(handles.colorbar_tag,'Value',0);

data.colorbar_handle=adjust_colorbar...
   (data.colorbar_handle,data.original_axes_position_bar,...
   data.original_axes_position_nobar,handles,handles.axes);

data.colorbar_handle_force=adjust_colorbar(data.colorbar_handle_force,...
   data.original_axes_position_bar_force,...
   data.original_axes_position_nobar_force,handles,handles.force_axes_tag);

set(handles.figure1,'UserData',data);

% --- Executes on slider movement.
function x_length_tag_Callback(hObject, eventdata, handles)
% hObject    handle to x_length_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'Value') returns position of slider
%        get(hObject,'Min') and get(hObject,'Max') to determine range of slider

[v,~]=get_normalized_length(hObject);
set(hObject,'Value',v);
set(handles.x_length_value,'String',sprintf('%3.1f',v));

% --- Executes during object creation, after setting all properties.
function x_length_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to x_length_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: slider controls usually have a light gray background.
if isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor',[.9 .9 .9]);
end



function x_length_value_Callback(hObject, eventdata, handles)
% hObject    handle to x_length_value (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of x_length_value as text
%        str2double(get(hObject,'String')) returns contents of x_length_value as a double




% --- Executes during object creation, after setting all properties.
function x_length_value_CreateFcn(hObject, eventdata, handles)
% hObject    handle to x_length_value (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end


% --- Executes on slider movement.
function y_length_tag_Callback(hObject, eventdata, handles)
% hObject    handle to y_length_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'Value') returns position of slider
%        get(hObject,'Min') and get(hObject,'Max') to determine range of slider

[v,~]=get_normalized_length(hObject);
set(hObject,'Value',v);
set(handles.y_length_value,'String',sprintf('%3.1f',v));

% --- Executes during object creation, after setting all properties.
function y_length_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to y_length_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: slider controls usually have a light gray background.
if isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor',[.9 .9 .9]);
end



function y_length_value_Callback(hObject, eventdata, handles)
% hObject    handle to y_length_value (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of y_length_value as text
%        str2double(get(hObject,'String')) returns contents of y_length_value as a double


% --- Executes during object creation, after setting all properties.
function y_length_value_CreateFcn(hObject, eventdata, handles)
% hObject    handle to y_length_value (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end

%------------------------------

%-----------------------------
function  [v, I] = get_normalized_length(h)
A = 1:5;
x = get(h, 'Value');
B = abs(x - A);
[~,I] = min(B);
v = A(I);


% --- Executes on button press in fixed_z_tag.
function fixed_z_tag_Callback(hObject, eventdata, handles)
% hObject    handle to fixed_z_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of fixed_z_tag

if get(hObject,'Value')==1  %if it one, then it was ZERO before
   set(handles.auto_z_tag,'Value',0);
else
   set(hObject,'Value',1);
end

% --- Executes on button press in auto_z_tag.
function auto_z_tag_Callback(hObject, eventdata, handles)
% hObject    handle to auto_z_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of auto_z_tag

if get(hObject,'Value')==1
   set(handles.fixed_z_tag,'Value',0);
else
   set(hObject,'Value',1);
end


% --- Executes on button press in add_label_tag.
function add_label_tag_Callback(hObject, eventdata, handles)
% hObject    handle to add_label_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of add_label_tag


% --- Executes on slider movement.
function contour_level_tag_Callback(hObject, eventdata, handles)
% hObject    handle to contour_level_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'Value') returns position of slider
%        get(hObject,'Min') and get(hObject,'Max') to determine range of slider

[v,~]= get_normalized_contour_level(hObject);
set(handles.contour_levels_value_tag,'String',sprintf('%d',v));
set(handles.contour_level_tag,'Value',v);

% --- Executes during object creation, after setting all properties.
function contour_level_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to contour_level_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: slider controls usually have a light gray background.
if isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor',[.9 .9 .9]);
end



function contour_levels_value_tag_Callback(hObject, eventdata, handles)
% hObject    handle to contour_levels_value_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of contour_levels_value_tag as text
%        str2double(get(hObject,'String')) returns contents of contour_levels_value_tag as a double


% --- Executes during object creation, after setting all properties.
function contour_levels_value_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to contour_levels_value_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end

%----------------------------
function  [v, I] = get_normalized_contour_level(h)
A = 5:55;
x = get(h, 'Value');
B = abs(x - A);
[~,I] = min(B);
v = A(I);


% --- Executes on button press in hidden_on_tag.
function hidden_on_tag_Callback(hObject, eventdata, handles)
% hObject    handle to hidden_on_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of hidden_on_tag

INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;
data=get(handles.figure1,'UserData');

if data.state ~= RUNNING && data.state ~= INIT
   update_plots(data, handles);
end



% --- Executes on button press in direct_solver_tag.
function direct_solver_tag_Callback(hObject, eventdata, handles)
% hObject    handle to direct_solver_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of direct_solver_tag

handles.output = hObject;
set(hObject,'Value',1);

set(handles.jacobiBtn, 'Value', 0);
set(handles.gaussSeidelBtn, 'Value', 0);
set(handles.SORBtn, 'Value', 0);

set(handles.gaussSeidelNaturalBtn, 'Enable', 'off');
set(handles.gaussSeidelNaturalBtn, 'Value', 1);

set(handles.gaussSeidelRedBlackBtn, 'Enable', 'off');
set(handles.SORoptimalBtn, 'Enable', 'off');
set(handles.SORuserBtn, 'Enable', 'off');

set(handles.conjugate_btn,'Value',0);
set(handles.steepest_btn,'Value',0);
set(handles.multi_btn,'Value',0);


% Update handles structure

guidata(hObject, handles);


% --- Executes on button press in grid_point_tag.
function grid_point_tag_Callback(hObject, eventdata, handles)
% hObject    handle to grid_point_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of grid_point_tag


function x_coord_tag_Callback(hObject, eventdata, handles)
% hObject    handle to x_coord_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of x_coord_tag as text
%        str2double(get(hObject,'String')) returns contents of x_coord_tag as a double


% --- Executes during object creation, after setting all properties.
function x_coord_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to x_coord_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end



function y_coord_tag_Callback(hObject, eventdata, handles)
% hObject    handle to y_coord_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of y_coord_tag as text
%        str2double(get(hObject,'String')) returns contents of y_coord_tag as a double


% --- Executes during object creation, after setting all properties.
function y_coord_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to y_coord_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end

% --- Executes on selection change in colormap_tag.
function colormap_tag_Callback(hObject, eventdata, handles)
% hObject    handle to colormap_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: contents = cellstr(get(hObject,'String')) returns colormap_tag contents as cell array
%        contents{get(hObject,'Value')} returns selected item from colormap_tag

colormap(get_colormap_color_as_string(handles));

function name = get_colormap_color_as_string(handles)

contents = cellstr(get(handles.colormap_tag,'String'));
name = contents{get(handles.colormap_tag,'Value')};

% --- Executes during object creation, after setting all properties.
function colormap_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to colormap_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: popupmenu controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end

function A_condition_tag_Callback(hObject, eventdata, handles)
% hObject    handle to A_condition_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of A_condition_tag as text
%        str2double(get(hObject,'String')) returns contents of A_condition_tag as a double


% --- Executes during object creation, after setting all properties.
function A_condition_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to A_condition_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end


% --- Executes on button press in show_grid_tag.
function show_grid_tag_Callback(hObject, eventdata, handles)
% hObject    handle to show_grid_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of show_grid_tag

INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;

data = get(handles.figure1, 'UserData');

if data.state ~= RUNNING && data.state ~= INIT
   update_plots(data, handles);
end

% --- Executes on selection change in shading_tag.
function shading_tag_Callback(hObject, eventdata, handles)
% hObject    handle to shading_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: contents = cellstr(get(hObject,'String')) returns shading_tag contents as cell array
%        contents{get(hObject,'Value')} returns selected item from shading_tag

INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;

data = get(handles.figure1, 'UserData');

if data.state ~= RUNNING && data.state ~= INIT
   update_plots(data, handles);
end

%-------------------------
function shading_name = get_current_shading(handles)
contents = cellstr(get(handles.shading_tag,'String'));
shading_name=contents{get(handles.shading_tag,'Value')};


% --- Executes during object creation, after setting all properties.
function shading_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to shading_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: popupmenu controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end



% --- Executes on button press in colorbar_tag.
function colorbar_tag_Callback(hObject, eventdata, handles)
% hObject    handle to colorbar_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of colorbar_tag

INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;

data = get(handles.figure1, 'UserData');

if data.state ~= RUNNING && data.state ~= INIT
   data = update_plots(data, handles);
   set(handles.figure1, 'UserData',data);
else
   update_force_plot(handles);
end


%----------------------------
function nzmax = nzmax_for_sparse(data)
nzmax = data.nx*data.ny;

if data.west_bc_type==1
   nzmax = nzmax + data.ny;
end

if data.north_bc_type==1
   nzmax = nzmax + data.nx;
end

if data.east_bc_type==1
   nzmax = nzmax + data.nx;
end

if data.south_bc_type==1
   nzmax = nzmax + data.nx;
end


% --- Executes on button press in steepest_btn.
function steepest_btn_Callback(hObject, eventdata, handles)
% hObject    handle to steepest_btn (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of steepest_btn

set(hObject,'Value',1);

set(handles.jacobiBtn, 'Value', 0);
set(handles.gaussSeidelBtn, 'Value', 0);
set(handles.SORBtn, 'Value', 0);
set(handles.direct_solver_tag, 'Value', 0);

set(handles.gaussSeidelNaturalBtn, 'Enable', 'off');
set(handles.gaussSeidelNaturalBtn, 'Value', 1);

set(handles.gaussSeidelRedBlackBtn, 'Enable', 'off');
set(handles.SORoptimalBtn, 'Enable', 'off');
set(handles.SORuserBtn, 'Enable', 'off');

set(handles.conjugate_btn,'Value',0);
set(handles.multi_btn,'Value',0);


% --- Executes on button press in conjugate_btn.
function conjugate_btn_Callback(hObject, eventdata, handles)
% hObject    handle to conjugate_btn (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of conjugate_btn

set(hObject,'Value',1);

set(handles.jacobiBtn, 'Value', 0);
set(handles.gaussSeidelBtn, 'Value', 0);
set(handles.SORBtn, 'Value', 0);
set(handles.direct_solver_tag, 'Value', 0);

set(handles.gaussSeidelNaturalBtn, 'Enable', 'off');
set(handles.gaussSeidelNaturalBtn, 'Value', 1);

set(handles.gaussSeidelRedBlackBtn, 'Enable', 'off');
set(handles.SORoptimalBtn, 'Enable', 'off');
set(handles.SORuserBtn, 'Enable', 'off');

set(handles.steepest_btn,'Value',0);
set(handles.multi_btn,'Value',0);


function conj_drop_tag_Callback(hObject, eventdata, handles)
% hObject    handle to conj_drop_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of conj_drop_tag as text
%        str2double(get(hObject,'String')) returns contents of conj_drop_tag as a double


% --- Executes during object creation, after setting all properties.
function conj_drop_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to conj_drop_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end



function conjugate_tol_tag_Callback(hObject, eventdata, handles)
% hObject    handle to conjugate_tol_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of conjugate_tol_tag as text
%        str2double(get(hObject,'String')) returns contents of conjugate_tol_tag as a double


% --- Executes during object creation, after setting all properties.
function conjugate_tol_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to conjugate_tol_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end


% --- Executes on button press in multi_btn.
function multi_btn_Callback(hObject, eventdata, handles)
% hObject    handle to multi_btn (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of multi_btn

set(hObject,'Value',1);

set(handles.jacobiBtn, 'Value', 0);
set(handles.gaussSeidelBtn, 'Value', 0);
set(handles.SORBtn, 'Value', 0);
set(handles.direct_solver_tag, 'Value', 0);

set(handles.gaussSeidelNaturalBtn, 'Enable', 'off');
set(handles.gaussSeidelNaturalBtn, 'Value', 1);

set(handles.gaussSeidelRedBlackBtn, 'Enable', 'off');
set(handles.SORoptimalBtn, 'Enable', 'off');
set(handles.SORuserBtn, 'Enable', 'off');

set(handles.conjugate_btn,'Value',0);
set(handles.steepest_btn,'Value',0);


function multi_tol_tag_Callback(hObject, eventdata, handles)
% hObject    handle to multi_tol_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of multi_tol_tag as text
%        str2double(get(hObject,'String')) returns contents of multi_tol_tag as a double


% --- Executes during object creation, after setting all properties.
function multi_tol_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to multi_tol_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end


% --- Executes on selection change in mu1_tag.
function mu1_tag_Callback(hObject, eventdata, handles)
% hObject    handle to mu1_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: contents = cellstr(get(hObject,'String')) returns mu1_tag contents as cell array
%        contents{get(hObject,'Value')} returns selected item from mu1_tag


% --- Executes during object creation, after setting all properties.
function mu1_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to mu1_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: popupmenu controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end


% --- Executes on selection change in mu2_tag.
function mu2_tag_Callback(hObject, eventdata, handles)
% hObject    handle to mu2_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: contents = cellstr(get(hObject,'String')) returns mu2_tag contents as cell array
%        contents{get(hObject,'Value')} returns selected item from mu2_tag


% --- Executes during object creation, after setting all properties.
function mu2_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to mu2_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: popupmenu controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end


% --- Executes on selection change in multi_smooth_tag.
function multi_smooth_tag_Callback(hObject, eventdata, handles)
% hObject    handle to multi_smooth_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: contents = cellstr(get(hObject,'String')) returns multi_smooth_tag contents as cell array
%        contents{get(hObject,'Value')} returns selected item from multi_smooth_tag


% --- Executes during object creation, after setting all properties.
function multi_smooth_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to multi_smooth_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: popupmenu controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end



function steep_tol_tag_Callback(hObject, eventdata, handles)
% hObject    handle to steep_tol_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of steep_tol_tag as text
%        str2double(get(hObject,'String')) returns contents of steep_tol_tag as a double


% --- Executes during object creation, after setting all properties.
function steep_tol_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to steep_tol_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end
