function [timeOfFlight,err] = nma_getDeltaTimeFromDeltaNu(r_vec,v_vec,nu0,nu,mu)
%calculates time of flight for the orbit moving from nu0 to nu.

% This function calculates the delta time, or the time of flight, 
% for the orbit moving from nu0 to nu.
% It does this calculations for any one of the 4 conic sections.
% 
% 
% INPUT
%   r_vec : Vector. The probe position vector. An array of 3 numbers.
%   v_vec : Vector. The probe velosity vector. An array of 3 numbers.
%    nu0 : scalar. The initial polar angle nu. in RADIANS.
%    nu  : scalar. The final polar angle nu. in RADIANS.
%    mu  : Scalar. The gravitional parameter. 
 
% OUTPUT
%    timeOfFlight; scalar. The time of flight moving from nu0 to nu.
%
% Algorithm:  Use 'e' to determine which conic section. From that
% apply the correct equation to derive the time-of-flight.
% see Bates,Mueller,White book, pages 182-189.

% Author: Nasser Abbasi
% Version: 1.0

%change history:
%
% name   date    changes
%-----  ------   ---------
% nma   4/30/03  Initial v1.0

timeOfFlight=0;
err='';

o=nma_getOrbitParams(r_vec,v_vec,mu);

e=norm(o.e_vec);

if(e<0)
    error('invalid value for e. Can not be negative');
end

if(e==0)  % circle
    err='circle not supported yet with delta angle. use delta time';
    return;
elseif(e==1) %parabola
    err='parabola not supported yet with delta angle. use delta time';
elseif( e>0 && e<1) %ellips
    err='ellips not supported yet with delta angle. use delta time.';
else % hyparabola
    nu  = nma_normalizeAngle(nu);
    nu0 = nma_normalizeAngle(nu0);
    
    F0 = acosh( (e+cos(nu0)) / (1+e*cos(nu0)) );
    F  = acosh( (e+cos(nu)) / (1+e*cos(nu)) );
    timeOfFlight = sqrt( (-o.a)^3/mu ) *  (   (e*sinh(F) -F) - (e*sinh(F0) - F0)  );    
end

%%%%%%%%%%%%%%%%%%5
%
%
%%%%%%%%%%%%%%%%%%
function [angle]=nma_normalizeAngle(angle)
if(  angle>pi & angle<2*pi)
    angle= -angle;
end
   
    






