function varargout = nma_diffusion_1d(varargin)
%main GUI file for 1D diffusion solver

% NMA_DIFFUSION_1D M-file for nma_diffusion_1d.fig
%      NMA_DIFFUSION_1D, by itself, creates a new NMA_DIFFUSION_1D or raises the existing
%      singleton*.
%
%      H = NMA_DIFFUSION_1D returns the handle to a new NMA_DIFFUSION_1D or the handle to
%      the existing singleton*.
%
%      NMA_DIFFUSION_1D('CALLBACK',hObject,eventData,handles,...) calls the local
%      function named CALLBACK in NMA_DIFFUSION_1D.M with the given input arguments.
%
%      NMA_DIFFUSION_1D('Property','Value',...) creates a new NMA_DIFFUSION_1D or raises the
%      existing singleton*.  Starting from the left, property value pairs are
%      applied to the GUI before nma_diffusion_1d_OpeningFcn gets called.  An
%      unrecognized property name or invalid value makes property application
%      stop.  All inputs are passed to nma_diffusion_1d_OpeningFcn via varargin.
%
%      *See GUI Options on GUIDE's Tools menu.  Choose "GUI allows only one
%      instance to run (singleton)".
%
% See also: GUIDE, GUIDATA, GUIHANDLES
% Copy right: Nasser M. Abbasi
% part of my matlab toolbox GUI apps, solves the 1D parabolic PDE
%
% Free to use for academic and research purposes.

% Edit the above text to modify the response to help nma_diffusion_1d

% Last Modified by GUIDE v2.5 11-Feb-2012 03:29:07

% Begin initialization code - DO NOT EDIT
gui_Singleton = 1;
gui_State = struct('gui_Name',       mfilename, ...
   'gui_Singleton',  gui_Singleton, ...
   'gui_OpeningFcn', @nma_diffusion_1d_OpeningFcn, ...
   'gui_OutputFcn',  @nma_diffusion_1d_OutputFcn, ...
   'gui_LayoutFcn',  [] , ...
   'gui_Callback',   []);
if nargin && ischar(varargin{1})
   gui_State.gui_Callback = str2func(varargin{1});
end

if nargout
   [varargout{1:nargout}] = gui_mainfcn(gui_State, varargin{:});
else
   gui_mainfcn(gui_State, varargin{:});
end
% End initialization code - DO NOT EDIT


% --- Executes just before nma_diffusion_1d is made visible.
function nma_diffusion_1d_OpeningFcn(hObject, eventdata, handles, varargin)
% This function has no output args, see OutputFcn.
% hObject    handle to figure
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
% varargin   command line arguments to nma_diffusion_1d (see VARARGIN)


% Choose default command line output for nma_diffusion_1d
handles.output = hObject;

%states that this program can be in at any one moment
INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;

set(handles.figure1, 'UserData',[]);
set(handles.figure1,'Name','1-D parabolic PDE solver, By Nasser M. Abbasi');

[a,map]=imread('nma_poisson_play_black.png');
[r,c,d]=size(a);
x=ceil(r/85);
y=ceil(c/85);
g=a(1:x:end,1:y:end,:);
g(g==255)=5.5*255;
set(handles.run_tag,'CData',g)

[a,map]=imread('nma_poisson_reset.jpg');
[r,c,d]=size(a);
x=ceil(r/85);
y=ceil(c/85);
g=a(1:x:end,1:y:end,:);
g(g==255)=5.5*255;
set(handles.reset_btn,'CData',g)


data.state = INIT;

set(handles.figure1, 'UserData',data);

% Update handles structure
guidata(hObject, handles);

run_tag_Callback(hObject, eventdata, handles);
reset_btn_Callback(hObject, eventdata, handles);
set(handles.max_t_tag,'String',1);



% UIWAIT makes nma_diffusion_1d wait for user response (see UIRESUME)
% uiwait(handles.figure1);

%-------------------------------
function [data,status] = read_initial_conditions(data,handles)

status = 0;

%initialize u first
data.u=zeros(data.N,1);

% read initial conditions
[data.u,status] = nma_evaluate_1D_function(get(handles.init_tag,'String'),data.X,'X');
if not(status)
   uiwait(errordlg('Invalid function for initial conditions, or synatx error',...
      'Bad Input', 'modal'));
   uicontrol(handles.init_tag);
   return
end

%read unit impulse
if get(handles.impulse_tag,'Value')==1
   
   [data.impulse_x,status]=nma_verify_valid_non_negative_numeric...
      (get(handles.impulse_at_tag,'String'),handles.impulse_at_tag,...
      'unit impulse location');
   if not(status)
      return;
   end
   
   if data.impulse_x>data.len || data.impulse_x<0
      errordlg('Invalid location of unit impulse, can not be outside',...
         'Bad Input', 'modal');
      status = 0;
      uicontrol(handles.init_tag);
      return
   end
   
   data.impulse_x = round(data.impulse_x/data.h)+1;
   set(handles.impulse_at_tag,'String',sprintf('%3.3f',data.X(data.impulse_x)));
   
   data.u(data.impulse_x)=data.u(data.impulse_x)+1;
end

%read step function
if get(handles.step_tag,'Value')==1
   
   [data.step_shift,status]=nma_verify_valid_non_negative_numeric...
      (get(handles.step_shift_tag,'String'),handles.step_shift_tag,...
      'unit step shift must be non-negative number');
   if not(status)
      return;
   end
   
   data.ending_index_of_unit_step = round(data.step_shift/data.h)+1;
   data.step_shift = (data.ending_index_of_unit_step-1)*data.h;
   if data.step_shift>data.len || data.step_shift<0
      uiwait(errordlg('Invalid shift of unit step, can not be outside',...
         'Bad Input', 'modal'));
      status = 0;
      uicontrol(handles.step_shift_tag);
      return
   end
   
   set(handles.step_shift_tag,'String',data.step_shift);
   
   [data.step_amplitude,status]=nma_verify_valid_numeric...
      (get(handles.step_amp_tag,'String'),handles.step_amp_tag,...
      'unit step amplitude must be a number');
   if not(status)
      return;
   end
   
   data.u(1:data.ending_index_of_unit_step )=...
      data.u(1:data.ending_index_of_unit_step )+data.step_amplitude;
end


%read unit rectangular pulse
if get(handles.rect_impulse_tag,'Value')==1
   
   %read and verify center of rect pulse
   [data.rect_at,status]=nma_verify_valid_non_negative_numeric...
      (get(handles.rect_at_tag,'String'),handles.rect_at_tag,...
      'unit rectangle center must be non-negative number');
   if not(status)
      return;
   end
   
   if data.rect_at == 0 || data.rect_at==data.len
      uiwait(errordlg('center of rectangle pulse can not be on the edge',...
         'Bad Input', 'modal'));
      status = 0;
      uicontrol(handles.rect_at_tag);
      return
   end
   
   data.rect_center_index = round(data.rect_at/data.h)+1;
   data.rect_center = (data.rect_center_index-1)*data.h;
   if data.rect_center>data.len || data.rect_center <0
      uiwait(errordlg('center of rectangle pulse can not be outside',...
         'Bad Input', 'modal'));
      status = 0;
      uicontrol(handles.handles.rect_at_tag);
      return
   end
   
   set(handles.rect_at_tag,'String',data.rect_center );
   
   %read and verify width of rect pulse
   [data.rect_width,status]=nma_verify_valid_positive_numeric...
      (get(handles.rect_width_tag,'String'),handles.rect_width_tag,...
      'unit rectangle width must be positive number');
   if not(status)
      return;
   end
   
   if data.rect_width>data.len
      uiwait(errordlg('unit rectangle width can not be larger than length',...
         'Bad Input', 'modal'));
      status = 0;
      uicontrol(handles.rect_width_tag);
      return
   end
   
   data.starting_index_of_unit_rect = ...
      round((data.rect_at-data.rect_width/2)/data.h)+1;
   data.starting_unit_rect = (data.starting_index_of_unit_rect-1)*data.h;
   if data.starting_unit_rect>data.len || data.starting_unit_rect<0
      uiwait(errordlg('unit rectangle can not fit inside domain, change center or width',...
         'Bad Input', 'modal'));
      status = 0;
      uicontrol(handles.rect_width_tag);
      return
   end
   
   data.ending_index_of_unit_rect = ...
      round((data.rect_at+data.rect_width/2)/data.h)+1;
   data.ending_unit_rect = (data.ending_index_of_unit_rect-1)*data.h;
   if data.ending_unit_rect>data.len || data.ending_unit_rect<0
      uiwait(errordlg('unit rectangle too wide to fit',...
         'Bad Input', 'modal'));
      status = 0;
      uicontrol(handles.rect_width_tag);
      return
   end
   
   set(handles.rect_width_tag,'String',data.rect_width);
   
   data.u(data.starting_index_of_unit_rect:data.ending_index_of_unit_rect )=...
      data.u(data.starting_index_of_unit_rect:data.ending_index_of_unit_rect )+1;
end

data.u_initial = data.u;
data.ymin=min(data.u); data.ymax=max(data.u);
if data.ymax==data.ymin
   data.ymax=0.001;
end
if data.ymin>0
   data.ymin=0;
end

status = 1;

%------------------------------
function [data,status] = read_source_terms(data,handles)


%read Q(t), external impulse source
data.add_source_location=true;

data.Q_string= strtrim(get(handles.Q_tag,'String'));
if strcmp(data.Q_string, '0') || strcmp(data.Q_string, '')
   data.Q_string = '0.*t';
   data.add_source_location=false;
   data.Q_x=0;
end
data.Q_string = ['@(t)' data.Q_string];

try
   data.Q = str2func(data.Q_string);
   data.Q(.1);
catch ME
   uiwait(errordlg('Invalid Q(t) function, check for valid Matlab synatx...',...
      'Bad Input', 'modal'));
   status = 0;
   uicontrol(handles.Q_tag);
   return
end

if data.add_source_location
   [data.Q_x,status]=nma_verify_valid_non_negative_numeric(...
      get(handles.Q_x_tag,'String'),handles.Q_x_tag,'Location of Q(t)');
   if not(status)
      return;
   end
   
   if data.Q_x>data.len
      uiwait(errordlg('Invalid location of Q(t), must be to be internal source',...
         'Bad Input', 'modal'));
      status = 0;
      uicontrol(handles.init_tag);
      return
   end
   
   if data.Q_x==data.len
      uiwait(errordlg('Location of Q(t) can not be located at right edge, must be internal',...
         'Bad Input', 'modal'));
      status = 0;
      uicontrol(handles.init_tag);
      return
   end
   
   if data.Q_x==0
      uiwait(errordlg('Location of Q(t) can not be located at left edge, must be internal',...
         'BadInput', 'modal'));
      status = 0;
      uicontrol(handles.init_tag);
      return
   end
   
   
   data.Q_x = round(data.Q_x/data.h)+1;
   if data.Q_x==1
      data.Q_x=data.Q_x+1;
   else
      if data.Q_x==data.N
         data.Q_x=data.Q_x-1;
      end
   end
   
   set(handles.Q_x_tag,'String',sprintf('%3.3f',data.X(data.Q_x)));
end


%add second source, the general source function, above was an impulse only
data.general_source_present = true;
data.G= strtrim(get(handles.general_source_tag,'String'));
if strcmp(data.G, '0') || strcmp(data.G, '')
   data.G = '0.*t';
   data.general_source_present = false;
end
data.G = ['@(X,t)' data.G];

try
   data.G = str2func(data.G);
   data.G(.5,0);
catch ME
   uiwait(errordlg('Invalid general_source function, check for valid Matlab synatx...',...
      'Bad Input', 'modal'));
   status = 0;
   uicontrol(handles.general_source_tag);
   return
end

% [Q,status] = nma_evaluate_1D_function(get(handles.Q_tag,'String'),X,'X');
% if not(status)
%    errordlg('Invalid function for Q(x,t), or synatx error',...
%       'Bad Input', 'modal');
%    uicontrol(handles.Q_tag);
%    return
% end

status = 1;

%-------------------------------------------
function [data,status] = read_boundary_conditions(data,handles)

data.u0='';
data.uL='';
data.neumann_left='';
data.neumann_right='';

if get(handles.periodic_bc_tag,'Value')==1
   status = 1;
   return;
end

if get(handles.dir_left_tag,'Value')==0 && ...
      get(handles.neumann_left_tag,'Value')==0
   uiwait(errordlg('No boundary conditions specified for left end...',...
      'Bad Input', 'modal'));
   status = 0;
   uicontrol(handles.dir_left_tag);
   return
end

if get(handles.dir_right_tag,'Value')==0 && ...
      get(handles.neumann_right_tag,'Value')==0
   uiwait(errordlg('No boundary conditions specified for right end...',...
      'Bad Input', 'modal'));
   status = 0;
   uicontrol(handles.dir_right_tag);
   return
end

if get(handles.dir_left_tag,'Value')==1
   data.u0= strtrim(get(handles.u0_tag,'String'));
   
   if strcmp(data.u0, '0') || strcmp(data.u0, '')
      data.u0 = '0.*t';
   end
   data.u0 = ['@(t)' data.u0];
   try
      data.u0 = str2func(data.u0);
      data.u0(0);
   catch ME
      uiwait(errordlg('Invalid left side dirichlet function, check for valid Matlab synatx...',...
         'Bad Input', 'modal'));
      status = 0;
      uicontrol(handles.u0_tag);
      return
   end
else %if neumann at left side, the u(0) comes from initial condition
   
   data.neumann_left= strtrim(get(handles.neumann_left_string,'String'));
   if strcmp(data.neumann_left, '0') || strcmp(data.neumann_left, '')
      data.neumann_left = '0.*t';
   end
   data.neumann_left = ['@(t)' data.neumann_left];
   try
      data.neumann_left = str2func(data.neumann_left);
      data.neumann_left(0);
   catch ME
      uiwait(errordlg('Invalid left side Neumman function, check for valid Matlab synatx...',...
         'Bad Input', 'modal'));
      status = 0;
      uicontrol(handles.neumann_left_string);
      return
   end
end


if get(handles.dir_right_tag,'Value')==1
   data.uL= strtrim(get(handles.uL_tag,'String'));
   
   if strcmp(data.uL, '0') || strcmp(data.uL, '')
      data.uL = '0.*t';
   end
   data.uL = ['@(t)' data.uL];
   try
      data.uL = str2func(data.uL);
      data.uL(0);
   catch ME
      uiwait(errordlg('Invalid right side dirichlet function, check for valid Matlab synatx...',...
         'Bad Input', 'modal'));
      status = 0;
      uicontrol(handles.uL_tag);
      return
   end
else %if neumann at right side, the u(1) comes from initial condition
   data.neumann_right = strtrim(get(handles.neumann_right_string,'String'));
   if strcmp(data.neumann_right, '0') || strcmp(data.neumann_right, '')
      data.neumann_right = '0.*t';
   end
   data.neumann_right = ['@(t)' data.neumann_right];
   try
      data.neumann_right = str2func(data.neumann_right);
      data.neumann_right(0);
   catch ME
      uiwait(errordlg('Invalid right side Neumman function, check for valid Matlab synatx...',...
         'Bad Input', 'modal'));
      status = 0;
      uicontrol(handles.neumann_right_string);
      return
   end
end

status = 1;

%--------------------------------------
function [status, data] = init_run(handles)

INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;

data=[];

str = get(handles.D_tag,'String');
[data.D, status] = ...
   nma_verify_valid_positive_numeric(str, handles.D_tag, 'diffusion constant');
if not(status)
   return;
end

str = get(handles.d_term_tag,'String');
[data.d,status]=nma_verify_valid_numeric(str,handles.d_term_tag,'d coefficient');
if not(status)
   return;
end

str = get(handles.a_tag,'String');
[data.a,status]=nma_verify_valid_numeric(str,handles.a_tag,'a coefficient');
if not(status)
   return;
end

str = get(handles.delt_tag,'String');
[data.delt, status] = ...
   nma_verify_valid_positive_numeric(str, handles.delt_tag, 'time step');
if not(status)
   return;
end

str = get(handles.max_t_tag,'String');
[data.max_t, status] = ...
   nma_verify_valid_positive_numeric(str, handles.max_t_tag, 'how long to run simulation');
if not(status)
   return;
end

if data.max_t<data.delt
   uiwait(errordlg('time to run can not be smaller than time step',...
      'Bad Input', 'modal'));
   status = 0;
   uicontrol(handles.delt_tag);
   return
end


str = get(handles.h_tag,'String');
[data.h, status] = ...
   nma_verify_valid_positive_numeric(str, handles.h_tag, 'space step');
if not(status)
   return;
end

str = get(handles.len_tag,'String');
[data.len, status] = ...
   nma_verify_valid_positive_numeric(str, handles.len_tag, 'length');
if not(status)
   return;
end


if data.len<data.h
   uiwait(errordlg('length can not be smaller than space step h',...
      'Bad Input', 'modal'));
   status = 0;
   uicontrol(handles.len_tag);
   return
end

if data.len<2*data.h
   uiwait(errordlg('length is too small or h too large, try again...',...
      'Bad Input', 'modal'));
   status = 0;
   uicontrol(handles.len_tag);
   return
end

data.N = floor(data.len/data.h)+1;
data.len = (data.N-1)*data.h;

data.X=linspace(0,data.len,data.N);

[data,status] = read_initial_conditions(data,handles);
if not(status)
   return;
end
[data,status] = read_source_terms(data,handles);
if not(status)
   return
end

[data,status]= read_boundary_conditions(data,handles);
if not(status)
   return;
end

cla(handles.axes,'reset');


user_data = get(handles.figure1, 'UserData');
user_data.state = RUNNING;
set(handles.figure1, 'UserData',user_data);

data.stability_condition = (data.delt/data.d)*(data.a/4 +data.D/data.h^2 );

if get(handles.forward_centered_tag,'Value')==1
   if data.stability_condition>0.5
      set(handles.warning_tag,'Visible','on');
      set(handles.CN_ok_tag,'Visible','off');         
      set(handles.warning_diffusion_convection_tag,'Visible','off');
      set(handles.ok_tag,'Visible','off');
      set(handles.diffusion_convection_stable_tag,'Visible','off');
   else
      set(handles.warning_tag,'Visible','off');
      set(handles.ok_tag,'Visible','on');
      set(handles.CN_ok_tag,'Visible','off');         
      set(handles.diffusion_convection_stable_tag,'Visible','off');
      set(handles.warning_diffusion_convection_tag,'Visible','off');
      set(handles.diffusion_convection_stable_tag,'Visible','off');
   end
elseif get(handles.CN_tag,'Value')==1
   set(handles.warning_tag,'Visible','off');
   set(handles.ok_tag,'Visible','off');
   set(handles.CN_ok_tag,'Visible','on');   
   set(handles.warning_diffusion_convection_tag,'Visible','off');
   set(handles.diffusion_convection_stable_tag,'Visible','off');
elseif get(handles.forward_centered_convection_tag,'Value')==1
   set(handles.CN_ok_tag,'Visible','off');      
   set(handles.ok_tag,'Visible','off');   
   set(handles.warning_tag,'Visible','off');      
      
   v=data.a*data.delt/data.h;
   u=data.D*data.delt/data.h^2;
   if (2*u>1) || (v^2>2*u)
      set(handles.warning_diffusion_convection_tag,'Visible','on');
      set(handles.diffusion_convection_stable_tag,'Visible','off');
   else
      set(handles.diffusion_convection_stable_tag,'Visible','on');        
      set(handles.warning_diffusion_convection_tag,'Visible','off');
   end
end

if get(handles.dir_left_tag,'Value')==1
   if get(handles.dir_right_tag,'Value')==1
      data.k=1:data.N-2;
   else
      data.k=1:data.N-1;
   end
else  %left side is neumman,
   if get(handles.dir_right_tag,'Value')==1
      data.k=0:data.N-1;
   else
      data.k=0:data.N;
   end
end

status = 1;

%----------------------------------------------------
function [status, data] = init_run0(handles,is_plot_spectrum)

[status, data] = init_run(handles);
if not(status)
   enable_all(handles);
   return;
end

if is_plot_spectrum
   plot_spectrum(data,handles);
end

if get(handles.forward_centered_tag,'Value')==1
   set(handles.delt_over_h_tag,'String',...
      sprintf('%2.5f',data.stability_condition));
else
   set(handles.delt_over_h_tag,'String','N/A');
end


% --- Executes on button press in run_tag.
function run_tag_Callback(hObject, eventdata, handles)
% hObject    handle to run_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;

% if get(handles.neumann_left_tag,'Value')==1 && ...
%       get(handles.neumann_right_tag,'Value')==1
%    errordlg('Error, both ends can not have Neumman boundary conditions...',...
%       'Bad Input', 'modal');
%    uicontrol(handles.len_tag);
%    return
% end

disable_all(handles);

if get(handles.refine_tag,'Value')==1
   set(handles.h_tag,'String',0.5);
   set(handles.delt_tag,'String',0.5);
   set(handles.len_tag,'String',1);
end


[status, data] = init_run0(handles,true);
if not(status)
   enable_all(handles);
   return;
end

if get(handles.refine_tag,'Value')==1
   data=process_refinement_study(data,handles);
else
   if get(handles.CN_tag,'Value')==1
      data=process_CN(data,handles);
   elseif get(handles.forward_centered_tag,'Value')==1
      data=process_explicit(data,handles);
   else
      data=process_explicit_diffusion_convection(data,handles);
   end
end


enable_all(handles);

%------------------------
function A=lap1d_diffusion_convection(n,c,r1,r2)
e=ones(n,1);
B=[r1*e c*e r2*e];
A=spdiags(B,[-1 0 1],n,n);


%------------------------------------------
function data=process_explicit_diffusion_convection(data,handles)
% This function implements the solution of the diffusion-convection
% problem using explicit forward-time centered space discretization
%
% problem 4, Math 228B, Winter 2011, UC Davis
%
% INPUT:
% data: a structure that contains the problem
%        parameters
% handles: the Matlab handles structure
%
% OUTPUT:
% data: updated data structure of the problem being solved.
%
% By Nasser M. Abbasi
INIT = 1;


if get(handles.periodic_bc_tag,'Value')==0
   uiwait(errordlg('Diffusion-Convection solver only supports periodic boundary conditions at this time...',...
      'Bad Input', 'modal'));
   uicontrol(handles.periodic_bc_tag);
   return
end


v = data.a*data.delt/data.h;
u = data.D*data.delt/data.h^2;

cla(handles.Q_axes,'reset');

% Make the update matrix. See my HW 1 report for details
%A     = lap1d_diffusion_convection(data.N-1,1-2*u,u+v/2,u-v/2);
A     = lap1d_diffusion_convection(data.N,1-2*u,u+v/2,u-v/2);
%b     = zeros(data.N-1,1);  %fix for other B.C.
b     = zeros(data.N,1);  %fix for other B.C.
A(1,end-1) = data.delt*data.D/(data.d*data.h^2);
A(end,2) = data.delt*data.D/(data.d*data.h^2);

% initialize before start of iterative loop
f          = 0;
nsteps     = floor(data.max_t/data.delt);
fps        = get_fps(handles,nsteps);
first_time = true;
current_t  = 0;
k          = 0;
done       = false;


while not(done)
   
   %update plot if needed
   if f==fps || first_time
      first_time = false;
      f=0;
      fps=get_fps(handles,nsteps);
      data=update_plot(current_t,k,data,handles);
   end
   
   f=f+1;
   
   if get(handles.periodic_bc_tag,'Value')==1
      b(2:end-1)=data.delt/data.d*data.G(data.X(2:end-1),current_t)';
   end
   
   if data.add_source_location
      b(data.Q_x-1)= b(data.Q_x-1) + data.delt*data.Q(current_t)/data.d;
   end
   
   
   % apply the update matrix
   data.u = (A*data.u')' + b';
   
   % check if user clicked on RESET
   userdata = get(handles.figure1, 'UserData');
   if userdata.state == INIT
      return;
   end
   
   current_t = current_t + data.delt;
   k = k +1;
   
   if current_t >= data.max_t
      if current_t > data.max_t
         current_t = data.max_t;
         k = k - 1;
      end
      
      data=update_plot(current_t,k,data,handles);
      done = true;
   end
end

%-----------------------------
function data=process_refinement_study(data,handles)
% This function implements refinement study for HW1
% problem 1, Math 228B, Winter 2011, UC Davis
%
% INPUT:
% data: a structure that contains the problem
%        parameters
% handles: the Matlab handles structure
%
% OUTPUT:
% data: updated data structure of the problem being solved.
%
% By Nasser M. Abbasi

% set up initialization for the error table, such as headings
% and formating
INIT = 1;
titles ={'#','delt','h','ratio'};
fms    ={'d','.7f','.7f','.4e'};
wid       = 14;
fileID    = 1;

% use 8 runs, and allocate the table to store the error and ratios
N=8;
table=zeros(N,6); % #, t, h, ||u||, ||u-u_last||, ratio

% Initialize space and time step to 1. They will start at 1/2 in the
% loop below
h=1;
t=1;
SKIP=2;

for n=1:N
   
   % Simulatiously divide space step and time step.
   % to show second order in space and time for C-N,
   % only need to divide by 2, but to show the same
   % for Forward-Euler, need to divide space by 2, but
   % time by 4.
   
   if get(handles.forward_centered_tag,'Value')==1 %Forward-Euler
      h=h/2;
      t=t/4;
   else                                            % C-N
      h=h/2;
      t=t/2;
   end
   
   % the following is just updates for the GUI framework
   set(handles.h_tag,'String',h);
   set(handles.delt_tag,'String',t);
   [~, data] = init_run0(handles,false);
   
   % check which scheme we are using, and call it to solve the pde
   if get(handles.CN_tag,'Value')==1
      data=process_CN(data,handles);
   else
      data=process_explicit(data,handles);
   end
   
   % the solution now is stored in data, the problem structure. Make
   % a new entry in the error table for this iteration.
   table(n,1) = n;
   table(n,2) = data.delt;
   table(n,3) = data.h;
   
   table(n,4)= sqrt(data.h)*norm(data.u,2);  % use grid 2-norm
        
   last_h = h;         
        
   if n==1
      last_solution = data.u;
   else  
      
      last_grid_size = length(last_solution);
      current_solution = last_solution;
      
      for i=1:last_grid_size          
          ii=i*SKIP-1;
          current_solution(i)=data.u(ii);                        
      end
            
      current_error = last_solution - current_solution;
      last_solution = data.u;   
      table(n,5) = sqrt(last_h)*norm(  current_error(:) ,2 ); %error      
     
      if n==2
         table(n,6)=1;
      else
         table(n,6) = table(n-1,5)/table(n,5); %e ratio
      end
      [hd,bdy]=nma_format_matrix(titles,...
         [table(2:n,1) table(2:n,2) table(2:n,3) table(2:n,6)],wid,fms,fileID,true );
      
      set(0,'CurrentFigure',handles.figure1);
      set(handles.figure1, 'CurrentAxes',handles.axes_region);
      cla('reset');
      
      text(-.2,.60,bdy,'FontSize',6);
      set(handles.axes_region,'YTick',[]);
      set(handles.axes_region,'XTick',[]);
      text(-.2,.9,hd,'FontSize',6);
      title('result of refinement study');
      drawnow();
      
   end
   
   % Check is user click on RESET button. If not, keep running
   userdata = get(handles.figure1, 'UserData');
   if userdata.state == INIT
      return;
   end
   
end

% The refinment study is completed. Generate plots and error table and
% send to the GUI

set(0,'CurrentFigure',handles.figure1);
set(handles.figure1, 'CurrentAxes',handles.axes);
cla('reset');
set(0,'defaultaxesfontsize',8) ;

loglog(table(2:end,3),table(2:end,5),'-d');
xlabel('log(h)','FontSize',8);
title({'refinement study result, ';'log(h^2+delt^2) vs successive errors difference'},'FontSize',8);

ylabel('log(error norm)','FontSize',8);

grid on;

%------------------------
function A=lap1d(n,r1,r2)
%Sets up the 1-D sparse update matrix for use with
%HW1 math 228N

e=ones(n,1);
B=[-r2*e r1*e -r2*e];
A=spdiags(B,[-1 0 1],n,n);

%---------------------------
function fps=get_fps(handles,nsteps)
fps = get(handles.speed_tag,'Value');
fps=round(fps*nsteps/100);
if fps==0
   fps=1;
end

%----------------------------------
function Q = update_Q(Qh,loc,t,delt,d,N)
Q=zeros(N,1);
Q(loc)= (delt / d)*Qh(t+delt/2);


%---------------------------------------------
function data=process_CN(data,handles)
% This function implements C-N scheme to solve parabolic pde
% for both Dirichlet and Nuemann boundary conditions
%
% INPUT:
% data: a structure that contains the problem
%        parameters
% handles: the Matlab handles structure
%
% OUTPUT:
% data: updated data structure of the problem being solved.
%
% By Nasser M. Abbasi

% Initialization of GUI fields
INIT = 1;
cla(handles.Q_axes,'reset');

% setup the B matrix, the update matrix
%see my HW1, math228B, for detailed derivation of these factors
%for the C-N scheme

r1=1+(data.delt*data.D/(data.d*data.h^2))+(data.delt*data.a/(2*data.d));
r2=data.delt*data.D/(2*data.d*data.h^2);
r3=1-(data.delt*data.D/(data.d*data.h^2))-(data.delt*data.a/(2*data.d));
r4=data.delt/(2*data.d);

%create 1-D sparse matrix. Then update it depending on the boundary
A=lap1d(data.N-2,r1,r2);      
b=zeros(data.N,1);
      
% Now check what boundary conditions are given. Depending on the
% B.C. the update matrix is modified as needed. The logic below implements
% this part

if get(handles.neumann_left_tag,'Value')==1
   if get(handles.neumann_right_tag,'Value')==1 %both neumann      
      tmp1=zeros(data.N-2,1);  %left column
      tmp1(1)=-r2;
      A=[tmp1 A];
      
      tmp1=zeros(1,data.N-1); %top row
      tmp1(1)=r1;
      tmp1(2)=-2*r2;
      A=[tmp1;A];
      
      tmp1=zeros(data.N-1,1); %right column
      tmp1(end)=-r2;
      A=[A tmp1];
      
      tmp1=zeros(1,data.N); %bottom row
      tmp1(end)=r1;
      tmp1(end-1)=-2*r2;
      A=[A;tmp1];
      
      from=1; to=data.N;
   else %left neumman, right dirch
      tmp1=zeros(data.N-2,1);
      tmp1(1)=-r2;
      A=[tmp1 A];
      tmp1=zeros(1,data.N-1);
      tmp1(1)=r1;
      tmp1(2)=-2*r2;
      A=[tmp1;A];
      A=[A; zeros(1,data.N-1)];
      A=[A zeros(data.N,1)];
      A(end,end)=1;
      from=1; to=data.N;
   end
else %left dirc so far
   if get(handles.neumann_right_tag,'Value')==1 %left direch, right nuemman
      tmp1=zeros(data.N-2,1);
      tmp1(end)=-r2;
      A=[A tmp1];
      tmp1=zeros(1,data.N-1);
      tmp1(end-1)=-2*r2;
      tmp1(end)=r1;
      A=[A;tmp1];
      A=[zeros(1,data.N-1); A];
      A=[zeros(data.N,1) A];
      A(1,1)=1;
      from=2; to=data.N;
   else %left and right dirch.      
      from=2; to=data.N-1;
      A=[zeros(1,data.N-2); A];
      A=[A; zeros(1,data.N-2)];
      A=[zeros(data.N,1) A];
      A=[A zeros(data.N,1)];
      A(1,1)=1;
      A(end,end)=1;
   end
end

% The update matrix is now setup. Now start the loop and run until max
% time. Do some initializtions before

f          = 0;
nsteps     = floor(data.max_t/data.delt);
fps        = get_fps(handles,nsteps);
first_time = true;
current_t  = 0;
k          = 0;
done       = false;

while not(done)
   
   % make a plot, only when needed. User can change this.
   if f==fps || first_time
      first_time = false;
      f=0;
      fps=get_fps(handles,nsteps);
      data=update_plot(current_t,k,data,handles);
   end
   
   f=f+1;
   
   %build the b vector. This depends on boundary conditions 4 cases to
   %check for
   
   if get(handles.dir_left_tag,'Value')==1
      b(1) = data.u0(current_t+data.delt);
      b(2) =r2*(data.u0(current_t)+data.u0(current_t+data.delt))+...
         r3*data.u(2)+r2*data.u(3)+...
         r4*( data.G(data.X(2),current_t) + ...
         data.G(data.X(2),current_t+data.delt));
   end
   
   if get(handles.dir_right_tag,'Value')==1
      b(end)=data.uL(current_t+data.delt);
      b(end-1) = r2*(data.uL(current_t)+data.uL(current_t+data.delt))+...
         r3*data.u(end-1)+r2*data.u(end-2)+...
         r4*( data.G(data.X(end-1),current_t) + ...
         data.G(data.X(end-1),current_t+data.delt));
   end
   
   if get(handles.neumann_left_tag,'Value')==1
      b(1)=r3*data.u(1)+r2*2*data.h*(data.neumann_left(current_t)+...
         data.neumann_left(current_t+data.delt))+...
         2*r2*data.u(2);
   end
   
   if get(handles.neumann_right_tag,'Value')==1
      b(end)=r3*data.u(end)+r2*2*data.h*(data.neumann_right(current_t)+...
         data.neumann_right(current_t+data.delt))+...
         2*r2*data.u(end-1);
   end
          
   %Finish making the b vector. We allready set b(1) and b(end) above
   j=from+1:to-1;
   b(j) = r2*data.u(j-1)+r3*data.u(j)+r2*data.u(j+1)+...
      r4*( data.G(data.X(j),current_t) + ...
      data.G(data.X(j),current_t+data.delt));

   if data.add_source_location   
       b = b + update_Q(data.Q,data.Q_x,current_t,data.delt,data.d,data.N);
   end

   %All is done. Now solve for u_n+1  

   data.u = (A\b)';

   % check is solution is blowing up (need to find better way to do this)
   %    if max(abs(u))>20*max(abs(u_initial)) || any(isnan(u))
   %       errordlg(sprintf('solution blow up exceeded limit at t=%f sec, terminating simulation...',k*delt),...
   %          'Bad Input', 'modal');
   %       uicontrol(handles.D_tag);
   %       enable_all(handles);
   %       return
   %    end
   
   %check is user hit RESET button
   userdata = get(handles.figure1, 'UserData');
   if userdata.state == INIT
      return;
   end
   
   
   %update the time
   current_t = current_t + data.delt;
   k = k +1;
   
   %if reached end of run, make final plot we are done
   if current_t >= data.max_t
      if current_t > data.max_t
         current_t = data.max_t;
         k = k - 1;
      end
      data=update_plot(current_t,k,data,handles);
      done = true;
   end
end

%--------------------------------

function A = make_A_parabolic_1D_explicit(L,R,k,h,D,a,d,N)
% determines the matrix A in  u_n+1 = A * u_n for the pde
% d u_t - D u_xx + a u = 0
%
% INPUT
%  L: left boundary 1 for Dirch. 2 for Neumann, 3 for Periodic
%  R: right boundary 1 for Dirch. 2 for Neumann, 3 for periodic
%  k: time step
%  h: space step
%  D: diffusion constant
%  a: from the above pde
%  d: from the above pde
%  N: total number of grid points (including boundaries, the whole thing)
%  

r1 = (1- 2*k*D/(d*h^2) - a*k/d );
r2 = k*D/(d*h^2);

if L==1&&R==1
   A=lap1d(N-2,r1,-r2);   
   A=[zeros(1,N-2);A];
   A=[zeros(N-1,1) A];
   A=[A zeros(N-1,1)];
   A=[A;zeros(1,N)];
elseif L==1 && R==2
   A=lap1d(N-1,r1,-r2); 
   A=[zeros(1,N-1);A];
   A=[zeros(N,1) A];   
   A(end,end-1)=2*r2;
elseif L==2 && R==1
   A=lap1d(N-1,r1,-r2);   
   A=[A zeros(N-1,1)];
   A=[A;zeros(1,N)];
   A(1,2)=2*r2;      
elseif L==2 && R==2
   A=lap1d(N,r1,-r2);   
   A(1,2)=2*r2; 
   A(end,end-1)=2*r2;
else %both periodic
   A=lap1d(N,r1,-r2);   
   A(1,end-1)=r2; 
   A(end,2)=r2;   
end

%---------------------------------------------
function data=process_explicit(data,handles)
% This function implements forward Euler scheme to solve parabolic pde
% for both Dirichlet and Nuemann boundary conditions
%
% INPUT:
% data: a structure that contains the problem
%        parameters
% handles: the Matlab handles structure
%
% OUTPUT:
% data: updated data structure of the problem being solved.
%
% By Nasser M. Abbasi

% Initialization of GUI fields

INIT = 1;
r2 = data.delt*data.D/(data.d*data.h^2);
r3 = data.delt/data.d;

cla(handles.Q_axes,'reset');

% check what boundary conditions are given. Depending on the
% B.C. the update matrix is created as needed.

if get(handles.periodic_bc_tag,'Value')==1
   L=3; R=3;
else
   if get(handles.neumann_left_tag,'Value')==1
      if get(handles.neumann_right_tag,'Value')==1
         L=2;
         R=2;
      else 
         L=2;
         R=1;
      end
   else   
      if get(handles.neumann_right_tag,'Value')==1
         L=1;
         R=2;
      else  
         L=1;
         R=1;
      end
   end
end

A = make_A_parabolic_1D_explicit(...
   L,R,data.delt,data.h,data.D,data.a,data.d,data.N);
b = zeros(data.N,1);

% The update matrix is now setup. Now start the loop and run until max
% time. Do some initializtions before
f          = 0;
nsteps     = floor(data.max_t/data.delt);
fps        = get_fps(handles,nsteps);
first_time = true;
current_t  = 0;
k          = 0;
done       = false;

while not(done)
   
   % make a plot, only when needed. User can change this.
   if f==fps || first_time
      first_time = false;
      f=0;
      fps=get_fps(handles,nsteps);
      data=update_plot(current_t,k,data,handles);
   end
   
   f=f+1;
            
   if get(handles.periodic_bc_tag,'Value')==1
      b(1) = 0; b(end)=0;
      b(2:end-1) = r3*data.G(data.X(2:end-1),current_t)';
   else
      if get(handles.neumann_left_tag,'Value')==1
         b(1)  = 2*data.h*data.neumann_left(current_t)*r2;         
         if get(handles.neumann_right_tag,'Value')==1 %both Neumann
            b(end)= 2*data.h*data.neumann_right(current_t)*r2;
            b(2:end-1) = r3*data.G(data.X(2:end-1),current_t)';                     
         else %left Neumann, right dirch.
            b(end) = data.uL(current_t+data.delt);
            b(end) = 0;
            b(2:end-1) = r3*data.G(data.X(2:end-1),current_t)';
            b(end-1) = b(end-1) + r2*data.uL(current_t);
         end         
      else
         b(1)   = data.u0(current_t+data.delt);
         if get(handles.neumann_right_tag,'Value')==1 %left dirch. right N
            b(2:end-2) = r3*data.G(data.X(2:end-1),current_t)';
            b(2) = b(2) + r2*data.u0(current_t);
            b(end) = 2*data.h*r2*data.neumann_right(current_t);
         else %both dirich.
            b(end) = data.uL(current_t+data.delt);            
            b(2:end-1) = r3*data.G(data.X(2:end-1),current_t)';
            b(2) = b(2)+r2*data.u0(current_t);
            b(end-2) = b(end-2)+r2*data.uL(current_t);
         end
      end
   end

   if data.add_source_location  %FIX ME, check index matches b layout
      b(data.Q_x) = b(data.Q_x)+data.delt*data.Q(current_t)/data.d;
   end
   
   data.u = (A*data.u' + b)';

   %check if user clicked RESET
   userdata = get(handles.figure1, 'UserData');
   if userdata.state == INIT
      return;
   end
   
   %update time and check if max time reached.
   current_t = current_t + data.delt;
   k = k +1;
   
   if current_t >= data.max_t
      if current_t > data.max_t
         current_t = data.max_t;
         k = k - 1;
      end
      data=update_plot(current_t,k,data,handles);
      done = true;
   end
   
end
%------------------------------
function enable_all(handles)

set(handles.periodic_bc_tag,'Enable','on');
set(handles.forward_centered_convection_tag,'Enable','on');

set(handles.refine_tag,'Enable','on');

set(handles.step_amp_tag,'Enable','on');
set(handles.step_tag,'Enable','on');
set(handles.step_shift_tag,'Enable','on');


set(handles.impulse_at_tag,'Enable','on');
set(handles.impulse_tag,'Enable','on');
set(handles.rect_impulse_tag,'Enable','on');
set(handles.rect_width_tag,'Enable','on');
set(handles.rect_at_tag,'Enable','on');



set(handles.CN_tag,'Enable','on');
set(handles.test_case_tag,'Enable','on');

set(handles.solution_in_3d_tag,'Enable','on');
set(handles.general_source_tag,'Enable','on');

set(handles.d_term_tag,'Enable','on');
set(handles.a_tag,'Enable','on');


set(handles.run_tag,'Enable','on');
set(handles.forward_centered_tag,'Enable','on');
set(handles.delt_tag,'Enable','on');
set(handles.h_tag,'Enable','on');
set(handles.len_tag,'Enable','on');
set(handles.D_tag,'Enable','on');
set(handles.max_t_tag,'Enable','on');
set(handles.init_tag,'Enable','on');
set(handles.Q_tag,'Enable','on');
set(handles.Q_x_tag,'Enable','on');

set(handles.dir_left_tag,'Enable','on');
if get(handles.dir_left_tag,'Value')==1
   set(handles.u0_tag,'Enable','on');
end

set(handles.neumann_left_tag,'Enable','on');
if get(handles.neumann_left_tag,'Value')==1
   set(handles.neumann_left_string,'Enable','on');
end


set(handles.dir_right_tag,'Enable','on');
if get(handles.dir_right_tag,'Value')==1
   set(handles.uL_tag,'Enable','on');
end

set(handles.neumann_right_tag,'Enable','on');
if get(handles.neumann_right_tag,'Value')==1
   set(handles.neumann_right_string,'Enable','on');
end


%------------------------------
function disable_all(handles)

set(handles.periodic_bc_tag,'Enable','off');

set(handles.forward_centered_convection_tag,'Enable','off');

set(handles.refine_tag,'Enable','off');

set(handles.step_amp_tag,'Enable','off');
set(handles.step_tag,'Enable','off');
set(handles.step_shift_tag,'Enable','off');



set(handles.impulse_at_tag,'Enable','off');
set(handles.impulse_tag,'Enable','off');
set(handles.rect_impulse_tag,'Enable','off');
set(handles.rect_width_tag,'Enable','off');
set(handles.rect_at_tag,'Enable','off');


set(handles.solution_in_3d_tag,'Enable','off');
set(handles.general_source_tag,'Enable','off');
set(handles.test_case_tag,'Enable','off');

set(handles.CN_tag,'Enable','off');
set(handles.run_tag,'Enable','off');

set(handles.d_term_tag,'Enable','off');
set(handles.a_tag,'Enable','off');


set(handles.forward_centered_tag,'Enable','off');
set(handles.delt_tag,'Enable','off');
set(handles.h_tag,'Enable','off');
set(handles.len_tag,'Enable','off');
set(handles.D_tag,'Enable','off');
set(handles.max_t_tag,'Enable','off');
set(handles.init_tag,'Enable','off');
set(handles.Q_tag,'Enable','off');
set(handles.Q_x_tag,'Enable','off');

set(handles.dir_left_tag,'Enable','off');
if get(handles.dir_left_tag,'Value')==1
   set(handles.u0_tag,'Enable','off');
end

set(handles.neumann_left_tag,'Enable','off');
if get(handles.neumann_left_tag,'Value')==1
   set(handles.neumann_left_string,'Enable','off');
end


set(handles.dir_right_tag,'Enable','off');
if get(handles.dir_right_tag,'Value')==1
   set(handles.uL_tag,'Enable','off');
end

set(handles.neumann_right_tag,'Enable','off');
if get(handles.neumann_right_tag,'Value')==1
   set(handles.neumann_right_string,'Enable','off');
end

%--------------------------
function update_plot_in_3d(current_t,k,data,handles)

set(0,'CurrentFigure',handles.figure1);
set(handles.figure1, 'CurrentAxes',handles.axes);

%view(110,40);
view(40,45);

mesh(data.X,[current_t; current_t],[data.u ;data.u]);
t1=sprintf('solution $u(x,t)$ at t=%3.5f sec',current_t);
t2=sprintf('current time step = %d',k);
title({t1;t2},'interpreter','latex','Fontsize',9);
set(gca,'FontSize',6);

xlabel('x');
ylabel('time');
zlabel('u(x,t)');

%zlim([data.ymin data.ymax]);

drawnow();

hold on

%--------------------------------------------
function data=update_plot(current_t,k,data,handles )

set(0,'CurrentFigure',handles.figure1);
set(handles.figure1, 'CurrentAxes',handles.axes);

if get(handles.solution_in_3d_tag,'Value')==1
   update_plot_in_3d(current_t,k,data,handles);
   return;
end

hold off;
plot(data.X,data.u,'.-'); xlabel('x');
hold on;
plot(data.X,data.u_initial,'or');
legend('current','initial');

if data.add_source_location
   plot(data.X(data.Q_x),0,'x');
end

if max(data.u) > data.ymax
   data.ymax=max(data.u);
end

if min(data.u) < data.ymin
   data.ymin=min(data.u);
end

if data.ymin==0
   ylim([0 data.ymax+0.3*data.ymax]);
else
   if data.ymin>=0
      ylim([data.ymin data.ymax+0.3*data.ymax]);
   else
      ylim([data.ymin+0.2*data.ymin data.ymax+0.3*data.ymax]);
   end
end

if data.X(1)==0
   xlim([-0.1 data.X(data.N)+0.2*data.X(data.N)]);
else
   xlim([0.9*data.data.X(1) data.X(data.N)+0.2*data.X(data.N)]);
end

t1=sprintf('solution $u(x,t)$ at t=%3.5f sec',current_t);
t2=sprintf('current time step = %d',k);
title({t1;t2},'interpreter','latex','Fontsize',9);
set(gca,'FontSize',6);

grid on;

drawnow();

set(handles.figure1, 'CurrentAxes',handles.Q_axes);
plot(current_t,data.Q(current_t),'.');
title('Q(t) profile','Fontsize',7);
xlabel('t (sec)');
set(gca,'FontSize',6);
hold on;
drawnow();

set(handles.figure1, 'CurrentAxes',handles.g_axes);
plot(data.X,data.G(data.X,current_t),'-');
title('current profile of source g(x,t)','Fontsize',7);
xlabel('x'); ylabel('g(x,t)');
set(gca,'FontSize',6);
drawnow();

%---------------------------------------
function status = is_constant_1D_function(str)

r1=strfind(str,'X');
if isempty(r1)
   status=1;
else
   status=0;
end

function pause_tag_Callback(hObject, eventdata, handles)
% hObject    handle to pause_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of pause_tag as text
%        str2double(get(hObject,'String')) returns contents of pause_tag as a double


% --- Executes during object creation, after setting all properties.
function pause_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to pause_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end


% --- Executes on button press in reset_btn.
function reset_btn_Callback(hObject, eventdata, handles)
% hObject    handle to reset_btn (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

INIT = 1; RUNNING = 2; PAUSED = 3; CONVERGED=4;

data = get(handles.figure1, 'UserData');
data.state = INIT;
enable_all(handles);
set(handles.figure1, 'UserData',data);

%------------------------------------------------
function plot_spectrum(data,handles)

set(0,'CurrentFigure',handles.figure1);
set(handles.figure1, 'CurrentAxes',handles.axes_eig);

eig_A=2*data.D/data.h^2*(cos(pi*data.k*data.h/data.len)-1);

scatter(eig_A,zeros(length(eig_A),1),2,'r')
grid on;

t1=sprintf('$\\lambda$ spectrum of linear operator');
t2=sprintf('$h=%s$, max $\\lambda=%f$',rats(data.h),max(eig_A));
t3=sprintf('min $\\lambda=%f$',min(eig_A));
t4=sprintf('condition number $=%f$',max(abs(eig_A))/min(abs(eig_A)));

title({t1;t2;t3;t4},'interpreter','latex','Fontsize',8);
xlhand = get(gca,'xlabel');
set(xlhand,'String','re( $\lambda$ )','interpreter','latex','Fontsize',6);
ylabel('im( $\lambda$ )','interpreter','latex');
set(gca,'FontSize',6);


region=eig_A*data.delt;
set(handles.figure1, 'CurrentAxes',handles.axes_region);
scatter(region,zeros(length(eig_A),1),2,'r');
grid on;

t = linspace(0,2*pi,100);
h=-1;
k=0;
r=1;
x = r*cos(t)+h;
y = r*sin(t)+k;
hold on;
plot(x,y);
axis square;
t1=sprintf('$\\lambda \\Delta(t)$ location in absolute stability region');
t2=sprintf('fastest time scale=%f (sec)',1/max(abs(eig_A)));
t3=sprintf('slowest time scale=%f (sec)',1/min(abs(eig_A)));

title({t1;t2;t3},'interpreter','latex','Fontsize',8);
set(gca,'FontSize',6);
hold off;



function neumann_left_string_Callback(hObject, eventdata, handles)
% hObject    handle to neumann_left_string (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of neumann_left_string as text
%        str2double(get(hObject,'String')) returns contents of neumann_left_string as a double


% --- Executes during object creation, after setting all properties.
function neumann_left_string_CreateFcn(hObject, eventdata, handles)
% hObject    handle to neumann_left_string (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end



function neumann_right_string_Callback(hObject, eventdata, handles)
% hObject    handle to neumann_right_string (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of neumann_right_string as text
%        str2double(get(hObject,'String')) returns contents of neumann_right_string as a double


% --- Executes during object creation, after setting all properties.
function neumann_right_string_CreateFcn(hObject, eventdata, handles)
% hObject    handle to neumann_right_string (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end


% --- Executes on slider movement.
function speed_tag_Callback(hObject, eventdata, handles)
% hObject    handle to speed_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'Value') returns position of slider
%        get(hObject,'Min') and get(hObject,'Max') to determine range of slider

x=round(get(hObject,'Value'));

set(handles.pause_tag,'String',sprintf('%d',x));

% --- Executes during object creation, after setting all properties.
function speed_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to speed_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: slider controls usually have a light gray background.
if isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor',[.9 .9 .9]);
end


% --- Executes when selected object is changed in uipanel7.
function uipanel7_SelectionChangeFcn(hObject, eventdata, handles)
% hObject    handle to the selected object in uipanel7
% eventdata  structure with the following fields (see UIBUTTONGROUP)
%	EventName: string 'SelectionChanged' (read only)
%	OldValue: handle of the previously selected object or empty if none was selected
%	NewValue: handle of the currently selected object
% handles    structure with handles and user data (see GUIDATA)

if get(handles.dir_left_tag,'Value')==1
   set(handles.u0_tag,'Enable','on');
   set(handles.neumann_left_string,'Enable','off');
   set(handles.periodic_bc_tag,'Value',0);
else
   set(handles.u0_tag,'Enable','off');
   set(handles.neumann_left_string,'Enable','on');
end


% --- Executes when selected object is changed in uipanel8.
function uipanel8_SelectionChangeFcn(hObject, eventdata, handles)
% hObject    handle to the selected object in uipanel8
% eventdata  structure with the following fields (see UIBUTTONGROUP)
%	EventName: string 'SelectionChanged' (read only)
%	OldValue: handle of the previously selected object or empty if none was selected
%	NewValue: handle of the currently selected object
% handles    structure with handles and user data (see GUIDATA)

if get(handles.dir_right_tag,'Value')==1
   set(handles.uL_tag,'Enable','on');
   set(handles.neumann_right_string,'Enable','off');
   set(handles.periodic_bc_tag,'Value',0);
else
   set(handles.uL_tag,'Enable','off');
   set(handles.neumann_right_string,'Enable','on');
end



function Q_tag_Callback(hObject, eventdata, handles)
% hObject    handle to Q_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of Q_tag as text
%        str2double(get(hObject,'String')) returns contents of Q_tag as a double


% --- Executes during object creation, after setting all properties.
function Q_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to Q_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end



function Q_x_tag_Callback(hObject, eventdata, handles)
% hObject    handle to Q_x_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of Q_x_tag as text
%        str2double(get(hObject,'String')) returns contents of Q_x_tag as a double


% --- Executes during object creation, after setting all properties.
function Q_x_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to Q_x_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end

function delt_over_h_tag_Callback(hObject, eventdata, handles)
% hObject    handle to delt_over_h_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of delt_over_h_tag as text
%        str2double(get(hObject,'String')) returns contents of delt_over_h_tag as a double


% --- Executes during object creation, after setting all properties.
function delt_over_h_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to delt_over_h_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end


function a_tag_Callback(hObject, eventdata, handles)
% hObject    handle to a_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of a_tag as text
%        str2double(get(hObject,'String')) returns contents of a_tag as a double


% --- Executes during object creation, after setting all properties.
function a_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to a_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end



function d_term_tag_Callback(hObject, eventdata, handles)
% hObject    handle to d_term_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of d_term_tag as text
%        str2double(get(hObject,'String')) returns contents of d_term_tag as a double


% --- Executes during object creation, after setting all properties.
function d_term_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to d_term_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end


function general_source_tag_Callback(hObject, eventdata, handles)
% hObject    handle to general_source_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of general_source_tag as text
%        str2double(get(hObject,'String')) returns contents of general_source_tag as a double


% --- Executes during object creation, after setting all properties.
function general_source_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to general_source_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end


% --- Executes on button press in solution_in_3d_tag.
function solution_in_3d_tag_Callback(hObject, eventdata, handles)
% hObject    handle to solution_in_3d_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of solution_in_3d_tag


% --- Executes on selection change in test_case_tag.
function test_case_tag_Callback(hObject, eventdata, handles)
% hObject    handle to test_case_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: contents = cellstr(get(hObject,'String')) returns test_case_tag contents as cell array
%        contents{get(hObject,'Value')} returns selected item from test_case_tag

case_number = get(hObject,'Value');
switch case_number
   case 1,
      
      set(handles.forward_centered_convection_tag,'Enable','off');
       set(handles.periodic_bc_tag,'Value',0);
      
      set(handles.delt_tag,'String','0.05');
      set(handles.h_tag,'String','0.1');
      set(handles.len_tag,'String','2*pi');
      set(handles.D_tag,'String','0.11');
      set(handles.a_tag,'String','0');
      set(handles.d_term_tag,'String','1');
      set(handles.max_t_tag,'String','3.1');
      
      set(handles.dir_left_tag,'Value',1);
      set(handles.u0_tag,'Enable','on');
      set(handles.neumann_left_string,'Enable','off');
      set(handles.u0_tag,'String',0);
      
      set(handles.dir_right_tag,'Value',1);
      set(handles.uL_tag,'Enable','on');
      set(handles.neumann_right_string,'Enable','off');
      set(handles.uL_tag,'String',0);
      set(handles.init_tag,'String','sin(X)');
      set(handles.speed_tag,'Value',1);
      set(handles.pause_tag,'String','1');
      set(handles.Q_tag,'String','0');
      set(handles.general_source_tag,'String','0');
      set(handles.forward_centered_tag,'Value',1);
      set(handles.forward_centered_convection_tag,'Value',0);
      set(handles.CN_tag,'Value',0);
      
      set(handles.impulse_tag,'Value',0);
      set(handles.rect_impulse_tag,'Value',0);
      set(handles.refine_tag,'Value',0);
      
   case 2,
       set(handles.periodic_bc_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Enable','off');
      
      
      set(handles.delt_tag,'String','0.05');
      set(handles.h_tag,'String','0.1');
      set(handles.len_tag,'String','2*pi');
      set(handles.D_tag,'String','0.11');
      set(handles.a_tag,'String','0');
      set(handles.d_term_tag,'String','1');
      set(handles.max_t_tag,'String','20');
      
      set(handles.dir_left_tag,'Value',1);
      set(handles.u0_tag,'Enable','on');
      set(handles.neumann_left_string,'Enable','off');
      set(handles.u0_tag,'String',0);
      
      set(handles.dir_right_tag,'Value',1);
      set(handles.uL_tag,'Enable','on');
      set(handles.neumann_right_string,'Enable','off');
      set(handles.uL_tag,'String',0);
      set(handles.init_tag,'String','sin(X)');
      
      set(handles.speed_tag,'Value',2);
      set(handles.pause_tag,'String','2');
      
      set(handles.Q_tag,'String','0');
      set(handles.general_source_tag,'String','0');
      set(handles.forward_centered_tag,'Value',0);
      
      set(handles.CN_tag,'Value',1);
      set(handles.forward_centered_convection_tag,'Value',0);
      
      set(handles.impulse_tag,'Value',0);
      set(handles.rect_impulse_tag,'Value',0);
      set(handles.refine_tag,'Value',0);
      
   case 3,
       set(handles.periodic_bc_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Enable','off');
      
      
      set(handles.delt_tag,'String','0.05');
      set(handles.h_tag,'String','0.1');
      set(handles.len_tag,'String','2*pi');
      set(handles.D_tag,'String','0.11');
      set(handles.a_tag,'String','0');
      set(handles.d_term_tag,'String','1');
      set(handles.max_t_tag,'String','3.1');
      
      set(handles.dir_left_tag,'Value',1);
      set(handles.u0_tag,'Enable','on');
      set(handles.neumann_left_string,'Enable','off');
      set(handles.u0_tag,'String',0);
      
      set(handles.dir_right_tag,'Value',1);
      set(handles.uL_tag,'Enable','on');
      set(handles.neumann_right_string,'Enable','off');
      set(handles.uL_tag,'String',0);
      set(handles.init_tag,'String','sin(X)');
      set(handles.speed_tag,'Value',0);
      set(handles.pause_tag,'String','0');
      
      set(handles.Q_tag,'String','t');
      set(handles.Q_x_tag,'String','0.5');
      
      set(handles.general_source_tag,'String','0');
      set(handles.forward_centered_tag,'Value',1);
      set(handles.CN_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Value',0);
      
      set(handles.impulse_tag,'Value',0);
      set(handles.rect_impulse_tag,'Value',0);
      set(handles.refine_tag,'Value',0);
      
   case 4,
       set(handles.periodic_bc_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Enable','off');
      
      
      set(handles.delt_tag,'String','0.05');
      set(handles.h_tag,'String','0.1');
      set(handles.len_tag,'String','2*pi');
      set(handles.D_tag,'String','0.11');
      set(handles.a_tag,'String','0');
      set(handles.d_term_tag,'String','1');
      set(handles.max_t_tag,'String','2.3');
      
      set(handles.dir_left_tag,'Value',1);
      set(handles.u0_tag,'Enable','on');
      set(handles.neumann_left_string,'Enable','off');
      set(handles.u0_tag,'String',0);
      
      set(handles.dir_right_tag,'Value',1);
      set(handles.uL_tag,'Enable','on');
      set(handles.neumann_right_string,'Enable','off');
      set(handles.uL_tag,'String',0);
      set(handles.init_tag,'String','sin(4*X)');
      set(handles.speed_tag,'Value',0);
      set(handles.pause_tag,'String','0');
      set(handles.Q_tag,'String','0');
      set(handles.general_source_tag,'String','0');
      set(handles.forward_centered_tag,'Value',1);
      set(handles.forward_centered_convection_tag,'Value',0);
      set(handles.CN_tag,'Value',0);
      set(handles.impulse_tag,'Value',0);
      set(handles.rect_impulse_tag,'Value',0);
      set(handles.refine_tag,'Value',0);
      
   case 5,
       set(handles.periodic_bc_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Enable','off');
      
      
      set(handles.delt_tag,'String','0.0118');
      set(handles.h_tag,'String','0.05');
      set(handles.len_tag,'String','2*pi');
      set(handles.D_tag,'String','0.11');
      set(handles.a_tag,'String','0');
      set(handles.d_term_tag,'String','1');
      set(handles.max_t_tag,'String','1.1');
      
      set(handles.dir_left_tag,'Value',0);
      set(handles.u0_tag,'Enable','off');
      
      set(handles.neumann_left_tag,'Value',1);
      set(handles.neumann_left_string,'Enable','on');
      set(handles.neumann_left_string,'String','0');
      
      set(handles.dir_right_tag,'Value',1);
      set(handles.uL_tag,'Enable','on');
      set(handles.neumann_right_string,'Enable','off');
      set(handles.uL_tag,'String',0);
      
      set(handles.init_tag,'String','sin(2*X)');
      set(handles.speed_tag,'Value',0);
      set(handles.pause_tag,'String','0');
      set(handles.Q_tag,'String','0');
      set(handles.general_source_tag,'String','0');
      set(handles.forward_centered_tag,'Value',1);
      set(handles.forward_centered_convection_tag,'Value',0);
      set(handles.CN_tag,'Value',0);
      set(handles.impulse_tag,'Value',0);
      set(handles.rect_impulse_tag,'Value',0);
      set(handles.refine_tag,'Value',0);
      
   case 6,
       set(handles.periodic_bc_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Enable','off');
      
      set(handles.delt_tag,'String','0.01');
      set(handles.h_tag,'String','0.1');
      set(handles.len_tag,'String','2*pi');
      set(handles.D_tag,'String','0.06');
      set(handles.a_tag,'String','0');
      set(handles.d_term_tag,'String','1');
      set(handles.max_t_tag,'String','80');
      
      set(handles.dir_left_tag,'Value',0);
      set(handles.u0_tag,'Enable','off');
      
      set(handles.neumann_left_tag,'Value',1);      
      set(handles.neumann_left_string,'Enable','on');
      set(handles.neumann_left_string,'String','0');
      
      set(handles.dir_right_tag,'Value',0);
      set(handles.uL_tag,'Enable','off');
      
      set(handles.neumann_right_tag,'Value',1);
      set(handles.neumann_right_string,'Enable','on');
      set(handles.neumann_right_string,'String','0');
      
      set(handles.init_tag,'String','sin(X/2)');
      set(handles.speed_tag,'Value',2);
      set(handles.pause_tag,'String','2');
      set(handles.Q_tag,'String','0');
      set(handles.general_source_tag,'String','0');
      set(handles.forward_centered_tag,'Value',1);
      set(handles.forward_centered_convection_tag,'Value',0);
      set(handles.CN_tag,'Value',0);
      set(handles.impulse_tag,'Value',0);
      set(handles.rect_impulse_tag,'Value',0);
      set(handles.refine_tag,'Value',0);
      
   case 7,
       set(handles.periodic_bc_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Enable','off');
      
      set(handles.delt_tag,'String','0.01');
      set(handles.h_tag,'String','0.1');
      set(handles.len_tag,'String','2*pi');
      set(handles.D_tag,'String','0.2');
      set(handles.a_tag,'String','0');
      set(handles.d_term_tag,'String','1');
      set(handles.max_t_tag,'String','400');
      
      set(handles.dir_left_tag,'Value',0);
      set(handles.u0_tag,'Enable','off');
      
      set(handles.neumann_left_tag,'Value',1);
      set(handles.neumann_left_string,'Enable','on');
      set(handles.neumann_left_string,'String','0');
      
      set(handles.dir_right_tag,'Value',1);
      set(handles.uL_tag,'Enable','on');
      
      set(handles.neumann_right_string,'Enable','off');
      set(handles.uL_tag,'String','0');
      
      set(handles.init_tag,'String','sin(X/2)');
      set(handles.speed_tag,'Value',4);
      set(handles.pause_tag,'String','4');
      set(handles.Q_tag,'String','0');
      set(handles.general_source_tag,'String','0');
      set(handles.forward_centered_tag,'Value',1);
      set(handles.forward_centered_convection_tag,'Value',0);
      set(handles.CN_tag,'Value',0);
      set(handles.impulse_tag,'Value',0);
      set(handles.rect_impulse_tag,'Value',0);
      set(handles.refine_tag,'Value',0);
      
   case 8,
       set(handles.periodic_bc_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Enable','off');
      
      set(handles.delt_tag,'String','0.01');
      set(handles.h_tag,'String','0.1');
      set(handles.len_tag,'String','2*pi');
      set(handles.D_tag,'String','0.2');
      set(handles.a_tag,'String','150');
      set(handles.d_term_tag,'String','1');
      set(handles.max_t_tag,'String','0.25');
      
      set(handles.dir_left_tag,'Value',0);
      set(handles.u0_tag,'Enable','off');
      
      set(handles.neumann_left_tag,'Value',1);
      set(handles.neumann_left_string,'Enable','on');
      set(handles.neumann_left_string,'String','0');
      
      set(handles.dir_right_tag,'Value',1);
      set(handles.uL_tag,'Enable','on');
      
      
      set(handles.neumann_right_string,'Enable','off');
      set(handles.uL_tag,'String','0');
      
      
      set(handles.init_tag,'String','sin(X/2)');
      set(handles.speed_tag,'Value',0);
      set(handles.pause_tag,'String','0');
      set(handles.Q_tag,'String','0');
      set(handles.general_source_tag,'String','0');
      set(handles.forward_centered_tag,'Value',1);
      set(handles.forward_centered_convection_tag,'Value',0);
      set(handles.CN_tag,'Value',0);
      set(handles.impulse_tag,'Value',0);
      set(handles.rect_impulse_tag,'Value',0);
      set(handles.refine_tag,'Value',0);
      
   case 9,
       set(handles.periodic_bc_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Enable','off');
      
      set(handles.delt_tag,'String','0.01');
      set(handles.h_tag,'String','0.1');
      set(handles.len_tag,'String','2*pi');
      set(handles.D_tag,'String','10');
      set(handles.a_tag,'String','0');
      set(handles.d_term_tag,'String','1');
      set(handles.max_t_tag,'String','3');
      
      set(handles.dir_left_tag,'Value',0);
      set(handles.u0_tag,'Enable','off');
      
      set(handles.neumann_left_tag,'Value',1);
      set(handles.neumann_left_string,'Enable','on');
      set(handles.neumann_left_string,'String','sin(2*pi*12*t)');
      
      set(handles.dir_right_tag,'Value',1);
      set(handles.uL_tag,'Enable','on');
      
      
      set(handles.neumann_right_string,'Enable','off');
      set(handles.uL_tag,'String','0');
      
      
      set(handles.init_tag,'String','sin(X/2)');
      set(handles.speed_tag,'Value',1);
      set(handles.pause_tag,'String','1');
      set(handles.Q_tag,'String','0');
      set(handles.general_source_tag,'String','0');
      set(handles.forward_centered_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Value',0);
      set(handles.CN_tag,'Value',1);
      set(handles.impulse_tag,'Value',0);
      set(handles.rect_impulse_tag,'Value',0);
      set(handles.refine_tag,'Value',0);
      
   case 10,
       set(handles.periodic_bc_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Enable','off');
      
      set(handles.delt_tag,'String','0.01');
      set(handles.h_tag,'String','0.1');
      set(handles.len_tag,'String','2*pi');
      set(handles.D_tag,'String','8');
      set(handles.a_tag,'String','0');
      set(handles.d_term_tag,'String','1');
      set(handles.max_t_tag,'String','0.3');
      
      set(handles.dir_left_tag,'Value',0);
      set(handles.u0_tag,'Enable','off');
      
      set(handles.neumann_left_tag,'Value',1);
      set(handles.neumann_left_string,'Enable','on');
      set(handles.neumann_left_string,'String','sin(2*pi*12*t)');
      
      set(handles.dir_right_tag,'Value',0);
      set(handles.uL_tag,'Enable','off');
      
      set(handles.neumann_right_string,'Enable','on');
      set(handles.neumann_right_string,'String','sin(2*pi*12*t)');
      set(handles.neumann_right_tag,'Value',1);
      
      
      set(handles.init_tag,'String','sin(X/2)');
      set(handles.speed_tag,'Value',0);
      set(handles.pause_tag,'String','0');
      set(handles.Q_tag,'String','0');
      set(handles.general_source_tag,'String','0');
      set(handles.forward_centered_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Value',0);
      set(handles.CN_tag,'Value',1);
      
      set(handles.impulse_tag,'Value',0);
      set(handles.rect_impulse_tag,'Value',0);
      set(handles.refine_tag,'Value',0);
      
   case 11,
     
      set(handles.periodic_bc_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Enable','off');
      
      set(handles.delt_tag,'String','0.005');
      set(handles.h_tag,'String','0.01');
      set(handles.len_tag,'String','0.25');
      set(handles.D_tag,'String','0.001');
      set(handles.a_tag,'String','0');
      set(handles.d_term_tag,'String','1');
      set(handles.max_t_tag,'String','0.5');
      
      set(handles.dir_left_tag,'Value',1);
      set(handles.u0_tag,'Enable','on');
      set(handles.u0_tag,'String','0');
      
      set(handles.neumann_left_tag,'Value',0);
      set(handles.neumann_left_string,'Enable','off');
      set(handles.neumann_left_string,'String','0');
      
      set(handles.dir_right_tag,'Value',1);
      set(handles.uL_tag,'Enable','on');
      set(handles.uL_tag,'String','0');
      
      set(handles.neumann_right_string,'Enable','off');
      set(handles.neumann_right_tag,'Value',0);
      
      
      set(handles.init_tag,'String','0');
      set(handles.speed_tag,'Value',0);
      set(handles.pause_tag,'String','0');
      set(handles.Q_tag,'String','0');
      set(handles.general_source_tag,'String','0');
      set(handles.forward_centered_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Value',0);
      set(handles.CN_tag,'Value',1);
      
      set(handles.impulse_tag,'Value',1);
      set(handles.impulse_at_tag,'String','0.125');
      set(handles.rect_impulse_tag,'Value',0);
      
      set(handles.solution_in_3d_tag,'Value',1);
      set(handles.refine_tag,'Value',0);
      
   case 12,
       set(handles.periodic_bc_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Enable','off');
      
      set(handles.delt_tag,'String','0.01');
      set(handles.h_tag,'String','0.01');
      set(handles.len_tag,'String','0.1');
      set(handles.D_tag,'String','0.0001');
      set(handles.a_tag,'String','0');
      set(handles.d_term_tag,'String','1');
      set(handles.max_t_tag,'String','0.5');
      
      set(handles.dir_left_tag,'Value',0);
      set(handles.u0_tag,'Enable','off');
      
      set(handles.neumann_left_tag,'Value',1);
      set(handles.neumann_left_string,'Enable','on');
      set(handles.neumann_left_string,'String','0');
      
      set(handles.dir_right_tag,'Value',0);
      set(handles.uL_tag,'Enable','off');
      
      set(handles.neumann_right_string,'Enable','on');
      set(handles.neumann_right_tag,'Value',1);
      set(handles.neumann_right_string,'String','0');
      
      
      set(handles.init_tag,'String','1');
      set(handles.speed_tag,'Value',0);
      set(handles.pause_tag,'String','0');
      set(handles.Q_tag,'String','0');
      set(handles.general_source_tag,'String','0');
      set(handles.forward_centered_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Value',0);
      set(handles.CN_tag,'Value',1);
      
      set(handles.impulse_tag,'Value',1);
      set(handles.impulse_at_tag,'String','0.05');
      set(handles.rect_impulse_tag,'Value',0);
      
      set(handles.solution_in_3d_tag,'Value',1);
      
      set(handles.refine_tag,'Value',0);
      
      
   case 13,
       set(handles.periodic_bc_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Enable','off');
      
      set(handles.delt_tag,'String','0.1');
      set(handles.h_tag,'String','0.1');
      set(handles.len_tag,'String','1');
      set(handles.D_tag,'String','0.01');
      set(handles.a_tag,'String','0');
      set(handles.d_term_tag,'String','1');
      set(handles.max_t_tag,'String','1');
      
      set(handles.dir_left_tag,'Value',1);
      set(handles.u0_tag,'Enable','on');
      set(handles.u0_tag,'String','0');
      
      set(handles.neumann_left_tag,'Value',0);
      set(handles.neumann_left_string,'Enable','off');
      set(handles.neumann_left_string,'String','0');
      
      set(handles.dir_right_tag,'Value',1);
      set(handles.uL_tag,'Enable','on');
      set(handles.uL_tag,'String','0');
      
      set(handles.neumann_right_string,'Enable','off');
      set(handles.neumann_right_tag,'Value',0);
      set(handles.neumann_right_string,'String','0');
      
      set(handles.init_tag,'String','0');
      set(handles.speed_tag,'Value',0);
      set(handles.pause_tag,'String','0');
      set(handles.Q_tag,'String','0');
      set(handles.general_source_tag,'String','1-exp(-t)');
      set(handles.forward_centered_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Value',0);
      set(handles.CN_tag,'Value',1);
      
      set(handles.impulse_tag,'Value',0);
      set(handles.impulse_at_tag,'String','0.5');
      set(handles.rect_impulse_tag,'Value',0);
      
      set(handles.solution_in_3d_tag,'Value',0);
      set(handles.refine_tag,'Value',1);
      set(handles.refine_tag,'Enable','on');
      
      
   case 14,
       set(handles.periodic_bc_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Enable','off');
      
      set(handles.delt_tag,'String','0.1');
      set(handles.h_tag,'String','0.1');
      set(handles.len_tag,'String','1');
      set(handles.D_tag,'String','0.01');
      set(handles.a_tag,'String','0');
      set(handles.d_term_tag,'String','1');
      set(handles.max_t_tag,'String','1');
      
      set(handles.dir_left_tag,'Value',1);
      set(handles.u0_tag,'Enable','on');
      set(handles.u0_tag,'String','0');
      
      set(handles.neumann_left_tag,'Value',0);
      set(handles.neumann_left_string,'Enable','off');
      set(handles.neumann_left_string,'String','0');
      
      set(handles.dir_right_tag,'Value',1);
      set(handles.uL_tag,'Enable','on');
      set(handles.uL_tag,'String','0');
      
      set(handles.neumann_right_string,'Enable','off');
      set(handles.neumann_right_tag,'Value',0);
      set(handles.neumann_right_string,'String','0');
      
      set(handles.init_tag,'String','0');
      set(handles.speed_tag,'Value',0);
      set(handles.pause_tag,'String','0');
      set(handles.Q_tag,'String','0');
      set(handles.general_source_tag,'String','1-exp(-t)');
      set(handles.forward_centered_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Value',0);
      set(handles.CN_tag,'Value',1);
      
      set(handles.impulse_tag,'Value',0);
      set(handles.impulse_at_tag,'String','0.5');
      set(handles.rect_impulse_tag,'Value',0);
      
      set(handles.solution_in_3d_tag,'Value',0);
      set(handles.refine_tag,'Value',1);
      set(handles.refine_tag,'Enable','on');
      
      
   case 15, %unstable diffusion-convection
      
      set(handles.periodic_bc_tag,'Enable','on');
      set(handles.periodic_bc_tag,'Value',1);
      
      set(handles.forward_centered_convection_tag,'Enable','on');
      set(handles.CN_tag,'Value',0);
      set(handles.forward_centered_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Value',1);
      
      
      set(handles.delt_tag,'String','0.000625');
      set(handles.h_tag,'String','0.05');
      set(handles.len_tag,'String','1');
      set(handles.D_tag,'String','1');
      set(handles.a_tag,'String','80');
      set(handles.d_term_tag,'String','1');
      set(handles.max_t_tag,'String','1');
      
      set(handles.dir_left_tag,'Value',0);
      set(handles.u0_tag,'Enable','off');
      
      set(handles.neumann_left_tag,'Value',0);
      set(handles.neumann_left_string,'Enable','off');
      
      set(handles.dir_right_tag,'Value',0);
      set(handles.uL_tag,'Enable','off');
      
      set(handles.neumann_right_string,'Enable','off');
      set(handles.neumann_right_tag,'Value',0);
      
      set(handles.init_tag,'String','exp(-20*(X-0.5).^2)');
      set(handles.speed_tag,'Value',0);
      set(handles.pause_tag,'String','0');
      set(handles.Q_tag,'String','0');
      set(handles.general_source_tag,'String','0');
      
      
      set(handles.impulse_tag,'Value',0);
      set(handles.impulse_at_tag,'String','0.5');
      set(handles.rect_impulse_tag,'Value',0);
      set(handles.step_tag,'Value',0);
      
      set(handles.solution_in_3d_tag,'Value',0);
      set(handles.refine_tag,'Value',0);
      set(handles.refine_tag,'Enable','on');
      
      
      
   case 16, %stable diffusion-convection
      set(handles.periodic_bc_tag,'Enable','on');
      set(handles.periodic_bc_tag,'Value',1);
      
      set(handles.forward_centered_convection_tag,'Enable','on');
      set(handles.CN_tag,'Value',0);
      set(handles.forward_centered_tag,'Value',0);
      set(handles.forward_centered_convection_tag,'Value',1);
      
      
      set(handles.delt_tag,'String','0.001');
      set(handles.h_tag,'String','0.05');
      set(handles.len_tag,'String','1');
      set(handles.D_tag,'String','0.01');
      set(handles.a_tag,'String','1');
      set(handles.d_term_tag,'String','1');
      set(handles.max_t_tag,'String','1');
      
      set(handles.dir_left_tag,'Value',0);
      set(handles.u0_tag,'Enable','off');
      
      set(handles.neumann_left_tag,'Value',0);
      set(handles.neumann_left_string,'Enable','off');
      
      set(handles.dir_right_tag,'Value',0);
      set(handles.uL_tag,'Enable','off');
      
      set(handles.neumann_right_string,'Enable','off');
      set(handles.neumann_right_tag,'Value',0);
      
      set(handles.init_tag,'String','exp(-20*(X-0.5).^2)');
      set(handles.speed_tag,'Value',1);
      set(handles.pause_tag,'String','1');
      set(handles.Q_tag,'String','0');
      set(handles.general_source_tag,'String','0');
      
      
      set(handles.impulse_tag,'Value',0);
      set(handles.impulse_at_tag,'String','0.5');
      set(handles.rect_impulse_tag,'Value',0);
      set(handles.step_tag,'Value',0);
      
      set(handles.solution_in_3d_tag,'Value',0);
      set(handles.refine_tag,'Value',0);
      set(handles.refine_tag,'Enable','on');
                       
end

% --- Executes during object creation, after setting all properties.
function test_case_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to test_case_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: popupmenu controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end

% --- Outputs from this function are returned to the command line.
function varargout = nma_diffusion_1d_OutputFcn(hObject, eventdata, handles)
% varargout  cell array for returning output args (see VARARGOUT);
% hObject    handle to figure
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Get default command line output from handles structure
varargout{1} = handles.output;


% --- Executes on button press in forward_centered_tag.
function forward_centered_tag_Callback(hObject, eventdata, handles)
% hObject    handle to forward_centered_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of forward_centered_tag

if get(hObject,'Value')==1
   set(handles.CN_tag,'Value',0);
   set(handles.forward_centered_convection_tag,'Value',0);
else
   set(hObject,'Value',1);
end


% --- Executes on button press in CN_tag.
function CN_tag_Callback(hObject, eventdata, handles)
% hObject    handle to CN_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of CN_tag

if get(hObject,'Value')==1
   set(handles.forward_centered_tag,'Value',0);
   set(handles.forward_centered_convection_tag,'Value',0);
else
   set(hObject,'Value',1);
end

% --- Executes on button press in forward_centered_convection_tag.
function forward_centered_convection_tag_Callback(hObject, eventdata, handles)
% hObject    handle to forward_centered_convection_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of forward_centered_convection_tag


if get(hObject,'Value')==1
   set(handles.CN_tag,'Value',0);
   set(handles.forward_centered_tag,'Value',0);
else
   set(hObject,'Value',1);
end


function delt_tag_Callback(hObject, ~, handles)
% hObject    handle to delt_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of delt_tag as text
%        str2double(get(hObject,'String')) returns contents of delt_tag as a double


% --- Executes during object creation, after setting all properties.
function delt_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to delt_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end



function h_tag_Callback(hObject, eventdata, handles)
% hObject    handle to h_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of h_tag as text
%        str2double(get(hObject,'String')) returns contents of h_tag as a double


% --- Executes during object creation, after setting all properties.
function h_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to h_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end



function D_tag_Callback(hObject, eventdata, handles)
% hObject    handle to D_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of D_tag as text
%        str2double(get(hObject,'String')) returns contents of D_tag as a double


% --- Executes during object creation, after setting all properties.
function D_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to D_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end



function len_tag_Callback(hObject, eventdata, handles)
% hObject    handle to len_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of len_tag as text
%        str2double(get(hObject,'String')) returns contents of len_tag as a double


% --- Executes during object creation, after setting all properties.
function len_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to len_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end



function init_tag_Callback(hObject, eventdata, handles)
% hObject    handle to init_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of init_tag as text
%        str2double(get(hObject,'String')) returns contents of init_tag as a double


% --- Executes during object creation, after setting all properties.
function init_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to init_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end



function u0_tag_Callback(hObject, eventdata, handles)
% hObject    handle to u0_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of u0_tag as text
%        str2double(get(hObject,'String')) returns contents of u0_tag as a double


% --- Executes during object creation, after setting all properties.
function u0_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to u0_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end



function uL_tag_Callback(hObject, eventdata, handles)
% hObject    handle to uL_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of uL_tag as text
%        str2double(get(hObject,'String')) returns contents of uL_tag as a double


% --- Executes during object creation, after setting all properties.
function uL_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to uL_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end

function max_t_tag_Callback(hObject, eventdata, handles)
% hObject    handle to max_t_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of max_t_tag as text
%        str2double(get(hObject,'String')) returns contents of max_t_tag as a double


% --- Executes during object creation, after setting all properties.
function max_t_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to max_t_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end


% --- Executes on button press in impulse_tag.
function impulse_tag_Callback(hObject, eventdata, handles)
% hObject    handle to impulse_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of impulse_tag



function impulse_at_tag_Callback(hObject, eventdata, handles)
% hObject    handle to impulse_at_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of impulse_at_tag as text
%        str2double(get(hObject,'String')) returns contents of impulse_at_tag as a double


% --- Executes during object creation, after setting all properties.
function impulse_at_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to impulse_at_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end


% --- Executes on button press in rect_impulse_tag.
function rect_impulse_tag_Callback(hObject, eventdata, handles)
% hObject    handle to rect_impulse_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of rect_impulse_tag



function rect_at_tag_Callback(hObject, eventdata, handles)
% hObject    handle to rect_at_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of rect_at_tag as text
%        str2double(get(hObject,'String')) returns contents of rect_at_tag as a double


% --- Executes during object creation, after setting all properties.
function rect_at_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to rect_at_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end



function rect_width_tag_Callback(hObject, eventdata, handles)
% hObject    handle to rect_width_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of rect_width_tag as text
%        str2double(get(hObject,'String')) returns contents of rect_width_tag as a double


% --- Executes during object creation, after setting all properties.
function rect_width_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to rect_width_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end


% --- Executes on button press in step_tag.
function step_tag_Callback(hObject, eventdata, handles)
% hObject    handle to step_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of step_tag



function step_amp_tag_Callback(hObject, eventdata, handles)
% hObject    handle to step_amp_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of step_amp_tag as text
%        str2double(get(hObject,'String')) returns contents of step_amp_tag as a double


% --- Executes during object creation, after setting all properties.
function step_amp_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to step_amp_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end



function step_shift_tag_Callback(hObject, eventdata, handles)
% hObject    handle to step_shift_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of step_shift_tag as text
%        str2double(get(hObject,'String')) returns contents of step_shift_tag as a double


% --- Executes during object creation, after setting all properties.
function step_shift_tag_CreateFcn(hObject, eventdata, handles)
% hObject    handle to step_shift_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
   set(hObject,'BackgroundColor','white');
end


% --- Executes on button press in refine_tag.
function refine_tag_Callback(hObject, eventdata, handles)
% hObject    handle to refine_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of refine_tag


% --- Executes on button press in periodic_bc_tag.
function periodic_bc_tag_Callback(hObject, eventdata, handles)
% hObject    handle to periodic_bc_tag (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of periodic_bc_tag

if get(hObject,'Value')==1
   if get(handles.dir_left_tag,'Value')==1
      set(handles.dir_left_tag,'Value',0);
      set(handles.u0_tag,'Enable','off');
   else
      set(handles.neumann_left_tag,'Value',0);
      set(handles.neumann_left_string,'Enable','off');
   end
   
   if get(handles.dir_right_tag,'Value')==1
      set(handles.dir_right_tag,'Value',0);
      set(handles.uL_tag,'Enable','off');
   else
      set(handles.neumann_right_tag,'Value',0);
      set(handles.neumann_right_string,'Enable','off');
   end
else
   set(hObject,'Value',1);
end
