(* ::Package:: *)

(* ::Subtitle:: *)
(*Utility Functions*)


(* ::Section::Closed:: *)
(*Miscellaneous Functions*)


sim[u_] := Simplify[u];
fsim[u_] := FullSimplify[u]


dx[u_] := D[u,x];
sdx[u_] := Simplify[D[u,x]];
fsdx[u_] := FullSimplify[D[u,x]]


ix[u_] := Integrate[u,x];
six[u_] := Simplify[Integrate[u,x]];
fidx[u_] := FullSimplify[Integrate[u,x]]


Second[u_] := u[[2]]


(* Note: Clear[func] also eliminates 2-D display of functions like Integrate. *) 
ClearDownValues[func_Symbol] := (
  Unprotect[func];
  DownValues[func]={};
  Protect[func])


SetDownValues[func_Symbol,lst_List] := (
  Unprotect[func];
  DownValues[func]=Take[lst,Min[529,Length[lst]]];
  Scan[Function[ReplacePart[ReplacePart[#,#[[1,1]],1],SetDelayed,0]],Drop[lst,Min[529,Length[lst]]]];
  Protect[func])


(* MoveDownValues[func1,func2] moves func1's DownValues to func2, and deletes them from func1. *)
MoveDownValues[func1_Symbol,func2_Symbol] := Module[{lst},
  SetDownValues[func2,ReplaceAll[DownValues[func1],{func1->func2}]];
  ClearDownValues[func1]]


Map2[func_,lst1_,lst2_] :=
  Reap[Do[Sow[func[lst1[[i]],lst2[[i]]]],{i,Length[lst1]}]][[2,1]]


(* MapAnd[f,l] applies f to the elements of list l until False is return; else returns True *)
MapAnd[f_,lst_List] :=
  Catch[Scan[Function[If[f[#],Null,Throw[False]]],lst];True]

MapAnd[f_,lst_List,x_] :=
  Catch[Scan[Function[If[f[#,x],Null,Throw[False]]],lst];True]


(* HeadSameQ[u,f] returns True if the head of u equals v; else it returns False. *)
HeadSameQ[u_,f_Symbol] :=
  If[ListQ[u],
    Catch[Scan[Function[If[HeadSameQ[#,f],Null,Throw[False]]],u];True],
  Head[u]===f]


(* ::Section::Closed:: *)
(*Recognizer Functions*)


(* ::Subsubsection::Closed:: *)
(*Number Domains*)


Unprotect[IntegerQ];
IntegerQ[u_List] :=
  MapAnd[IntegerQ,u]
Protect[IntegerQ];


(* PositiveIntegerQ[u] returns True if u is a positive integer, else it returns False. *)
PositiveIntegerQ[u_] :=
  If[ListQ[u],
    MapAnd[PositiveIntegerQ,u],
  IntegerQ[u] && u>0]


(* ZeroQ[u] returns True if u is any 0; else returns False *)
ZeroQ[u_] :=
  If[ListQ[u],
    MapAnd[ZeroQ,u],
  PossibleZeroQ[u]]


(* NonzeroQ[u] returns True if u is not any 0, else it returns False. *)
NonzeroQ[u_] :=
  If[ListQ[u],
    MapAnd[NonzeroQ,u],
  Not[PossibleZeroQ[u]]]


(* RealNumericQ[u] returns True if u is a real numeric quantity, else returns False. *)
RealNumericQ[u_] :=
  NumericQ[u] && PossibleZeroQ[Im[N[u]]]


(* ImaginaryNumericQ[u] returns True if u is an imaginary numeric quantity, else returns False. *)
ImaginaryNumericQ[u_] :=
  NumericQ[u] && PossibleZeroQ[Re[N[u]]] && Not[PossibleZeroQ[Im[N[u]]]]


(* PositiveQ[u] returns True if u is a positive numeric quantity, else returns False. *)
PositiveQ[u_] :=
  If[ListQ[u],
    MapAnd[PositiveQ,u],
  RealNumericQ[u] && N[u]>0]


(* NonpositiveQ[u] returns True if u is a nonpositive numeric quantity, else returns False. *)
PositiveOrZeroQ[u_] :=
  If[ListQ[u],
    MapAnd[PositiveOrZeroQ,u],
  RealNumericQ[u] && N[u]>=0]


(* NegativeQ[u] returns True if u is a negative numeric quantity, else returns False. *)
NegativeQ[u_] :=
  If[ListQ[u],
    MapAnd[NegativeQ,u],
  RealNumericQ[u] && N[u]<0]


(* NegativeQ[u] returns True if u is a negative numeric quantity, else returns False. *)
NegativeOrZeroQ[u_] :=
  If[ListQ[u],
    MapAnd[NegativeOrZeroQ,u],
  RealNumericQ[u] && N[u]<=0]


(* ::Subsubsection::Closed:: *)
(*Number Types*)


(* FractionQ[u] returns True if u is an explicit fraction; else returns False. *)
FractionQ[u_] :=
  HeadSameQ[u,Rational]


(* RationalQ[u] returns True if u is an explicit integer or fraction; else returns False. *)
RationalQ[u_] :=
  If[ListQ[u],
    MapAnd[RationalQ,u],
  IntegerQ[u] || FractionQ[u]]


(* HalfIntegerQ[u] returns True if u is a fraction with a denominator of 2; else returns False *)
HalfIntegerQ[u_] :=
  If[ListQ[u],
    MapAnd[HalfIntegerQ,u],
  FractionQ[u] && Denominator[u]==2]


(* FractionOrNegativeQ[u] returns True if u is a fraction or negative number; else returns False *)
FractionOrNegativeQ[u_] :=
  If[ListQ[u],
    MapAnd[FractionOrNegativeQ,u],
  FractionQ[u] || IntegerQ[u] && u<0]


(* AbsurdNumberQ[u] returns True if u is a real-valued absurd number (a rational number, a
   positive rational number raised to a fractional power, or a product of absurd numbers); 
   else returns False. *)
AbsurdNumberQ[u_List] :=
  MapAnd[AbsurdNumberQ,u]

AbsurdNumberQ[u_^v_] :=
  RationalQ[u] && u>0 && FractionQ[v]

AbsurdNumberQ[u_*v_] :=
  AbsurdNumberQ[u] && AbsurdNumberQ[v]

AbsurdNumberQ[u_] :=
  RationalQ[u]


(* AlgebraicNumberQ[u] returns True if u is a real-valued algebraic number (a rational number,
   an algebraic number raised to an integer power, a positive algebraic number raised to a 
   fractional power, or a product or sum of algebraic numbers); else returns False. *)
AlgebraicNumberQ[u_] :=
  MapAnd[AlgebraicNumberQ,u] /;
ListQ[u]

AlgebraicNumberQ[u_^v_] :=
  AlgebraicNumberQ[u] && (IntegerQ[v] || PositiveQ[u] && FractionQ[v])

AlgebraicNumberQ[u_*v_] :=
  AlgebraicNumberQ[u] && AlgebraicNumberQ[v]

AlgebraicNumberQ[u_+v_] :=
  AlgebraicNumberQ[u] && AlgebraicNumberQ[v]

AlgebraicNumberQ[u_] :=
  RationalQ[u]


(* ::Subsubsection::Closed:: *)
(*Expression Types*)


FalseQ[u_] :=
  u===False


NotFalseQ[u_] :=
  u=!=False


SumQ[u_] :=
  Head[u]===Plus

NonsumQ[u_] :=
  Head[u]=!=Plus

ProductQ[u_] :=
  Head[u]===Times

PowerQ[u_] :=
  Head[u]===Power

IntegerPowerQ[u_] :=
  PowerQ[u] && IntegerQ[u[[2]]]

FractionalPowerQ[u_] :=
  PowerQ[u] && FractionQ[u[[2]]]

SqrtQ[u_] :=
  PowerQ[u] && u[[2]]===1/2

ExpQ[u_] :=
  PowerQ[u] && u[[1]]===E


LogQ[u_] :=
  Head[u]===Log


SinQ[u_] :=
  Head[u]===Sin

CosQ[u_] :=
  Head[u]===Cos

TanQ[u_] :=
  Head[u]===Tan

CotQ[u_] :=
  Head[u]===Cot

SecQ[u_] :=
  Head[u]===Sec

CscQ[u_] :=
  Head[u]===Csc


SinhQ[u_] :=
  Head[u]===Sinh

CoshQ[u_] :=
  Head[u]===Cosh

TanhQ[u_] :=
  Head[u]===Tanh

CothQ[u_] :=
  Head[u]===Coth

SechQ[u_] :=
  Head[u]===Sech

CschQ[u_] :=
  Head[u]===Csch


(* TrigQ[u] returns True if the head of u is a trig function; else returns False *)
TrigQ[u_] :=
  MemberQ[{Sin,Cos,Tan,Cot,Sec,Csc},Head[u]]

(* InverseTrigQ[u] returns True if the head of u is an inverse trig function; else returns False *)
InverseTrigQ[u_] :=
  MemberQ[{ArcSin,ArcCos,ArcTan,ArcCot,ArcSec,ArcCsc},Head[u]]

(* HyperbolicQ[u] returns True if the head of u is a trig function; else returns False *)
HyperbolicQ[u_] :=
  MemberQ[{Sinh,Cosh,Tanh,Coth,Sech,Csch},Head[u]]

(* InverseHyperbolicQ[u] returns True if the head of u is an inverse trig function; else returns False *)
InverseHyperbolicQ[u_] :=
  MemberQ[{ArcSinh,ArcCosh,ArcTanh,ArcCoth,ArcSech,ArcCsch},Head[u]]


CalculusFunctions={D,Integrate,Sum,Product,Int,Dif,Subst};

(* CalculusQ[u] returns True if the head of u is a calculus function; else returns False *)
CalculusQ[u_] :=
  MemberQ[CalculusFunctions,Head[u]]

CalculusFreeQ[u_,x_] :=
  If[AtomQ[u],
    True,
  If[CalculusQ[u] && u[[2]]===x || Head[u]===Pattern || Head[u]===Defer,
    False,
  Catch[Scan[Function[If[CalculusFreeQ[#,x],Null,Throw[False]]],u];True]]]


SubstQ[u_] :=
  Head[u]===Subst


(* ElementaryQ[u] returns True if u is an elementary expression; else returns False *)
ElementaryQ[u_] :=
  If[AtomQ[u],
    True,
  If[SumQ[u] || ProductQ[u] || PowerQ[u] || LogQ[u] || TrigQ[u] || InverseTrigQ[u] ||
      HyperbolicQ[u] || InverseHyperbolicQ[u],
    Catch[Scan[Function[If[ElementaryQ[#],Null,Throw[False]]],u];True],
  False]]


(* ::Subsubsection::Closed:: *)
(*Expression Domains*)


(* Real[u] returns True if u is a real-valued quantity, else returns False. *)
RealQ[u_] :=
  MapAnd[RealQ,u] /;
ListQ[u]

RealQ[u_] :=
  PossibleZeroQ[Im[N[u]]] /;
NumericQ[u]

RealQ[u_^v_] :=
  RealQ[u] && RealQ[v] && (IntegerQ[v] || PositiveOrZeroQ[u])  

RealQ[u_*v_] :=
  RealQ[u] && RealQ[v]

RealQ[u_+v_] :=
  RealQ[u] && RealQ[v]

RealQ[f_[u_]] :=
  If[MemberQ[{Sin,Cos,Tan,Cot,Sec,Csc,ArcTan,ArcCot,Erf},f],
    RealQ[u],
  If[MemberQ[{ArcSin,ArcCos},f],
    LE[-1,u,1],
  If[f===Log,
    PositiveOrZeroQ[u],
  False]]]

RealQ[u_] :=
  False


(* If u is not 0 and has a positive form, PosQ[u] returns True, else it returns False. *)
PosQ[u_] :=
  If[PossibleZeroQ[u],
    False,
  If[NumericQ[u],
    If[NumberQ[u],
      If[PossibleZeroQ[Re[u]],
        Im[u]>0,
      Re[u]>0],
    Module[{v=N[u]},
    If[PossibleZeroQ[Re[v]],
      Im[v]>0,
    Re[v]>0]]],
  Module[{v=Simplify[u]},
  If[NumericQ[v],
    PosQ[v],
  If[SumQ[v] || ProductQ[v],
    PosQ[First[v]],
  Not[MatchQ[v,-_]]]]]]]


(* If u is not 0 and has a negative form, NegQ[u] returns True, else it returns False. *)
NegQ[u_] :=
  If[PossibleZeroQ[u],
    False,
  If[NumericQ[u],
    If[NumberQ[u],
      If[PossibleZeroQ[Re[u]],
        Im[u]<0,
      Re[u]<0],
    Module[{v=N[u]},
    If[PossibleZeroQ[Re[v]],
      Im[v]<0,
    Re[v]<0]]],
  Module[{v=Simplify[u]},
  If[NumericQ[v],
    NegQ[v],
  If[SumQ[v] || ProductQ[v],
    NegQ[First[v]],
  MatchQ[v,-_]]]]]]


(* ::Section::Closed:: *)
(*Product Selector Functions*)


(* LeadFactor[u] returns the leading factor of u. *)
LeadFactor[u_] :=
  If[ProductQ[u],
    First[u],
  u]


(* LeadBase[u] returns the base of the leading factor of u. *)
LeadBase[u_] :=
  If[PowerQ[u],
    u[[1]],
  If[ProductQ[u],
    If[PowerQ[First[u]],
      First[u][[1]],
    First[u]],
  u]]


(* LeadDegree[u] returns the degree of the leading factor of u. *)
LeadDegree[u_] :=
  If[PowerQ[u],
    u[[2]],
  If[ProductQ[u],
    If[PowerQ[First[u]],
      First[u][[2]],
    1],
  1]]


(* RemainingFactors[u] returns the remaining factors of u. *)
RemainingFactors[u_] :=
  If[ProductQ[u],
    Rest[u],
  1]


(* ::Section::Closed:: *)
(*Symbolic Relational Operators*)


(* LT[u,v] returns True if u and v are real-valued numeric quantities and u<v, else returns False *)
LT[u_,v_] :=
  RealNumericQ[u] && RealNumericQ[v] && N[u]<N[v]

LT[u_,v_,w_] :=
  LT[u,v] && LT[v,w]


(* LE[u,v] returns True if u and v are real-valued numeric quantities and u<=v, else returns False *)
LE[u_,v_] :=
  RealNumericQ[u] && RealNumericQ[v] && N[u]<=N[v]

LE[u_,v_,w_] :=
  LE[u,v] && LE[v,w]


(* GT[u,v] returns True if u and v are real-valued numeric quantities and u>v, else returns False *)
GT[u_,v_] :=
  RealNumericQ[u] && RealNumericQ[v] && N[u]>N[v]

GT[u_,v_,w_] :=
  GT[u,v] && GT[v,w]


(* GE[u,v] returns True if u and v are real-valued numeric quantities and u>=v, else returns False *)
GE[u_,v_] :=
  RealNumericQ[u] && RealNumericQ[v] && N[u]>=N[v]

GE[u_,v_,w_] :=
  GE[u,v] && GE[v,w]


(* ::Section::Closed:: *)
(*Variable Dependence Functions*)


IndependentQ[u_,x_Symbol] :=
  FreeQ[u,x]


DependentQ[u_,x_Symbol] :=
  Not[FreeQ[u,x]]


(* IndependentFactors[u,x] returns the product of the factors of u free of x.
	Compare with the more active function ConstantFactor. *)
IndependentFactors[u_,x_Symbol] :=
  If[ProductQ[u],
    Map[Function[If[FreeQ[#,x],#,1]],u],
  If[FreeQ[u,x],
    u,
  1]]


(* DependentFactors[u,x] returns the product of the factors of u not free of x. *)
DependentFactors[u_,x_Symbol] :=
  If[ProductQ[u],
    Map[Function[If[FreeQ[#,x],1,#]],u],
  If[FreeQ[u,x],
    1,
  u]]


(* ::Section::Closed:: *)
(*Polynomial Functions*)


(* ::Subsubsection::Closed:: *)
(*Polynomial Recognizer Functions*)


Unprotect[PolynomialQ];
PolynomialQ[u_List,x_Symbol] :=
  MapAnd[PolynomialQ,u,x]
Protect[PolynomialQ];


(* If u(x) is equivalent to an expression of the form a+b*x, LinearQ[u,x] returns True;
	else it returns False. *)
LinearQ[u_,x_Symbol] :=
  PolynomialQ[u,x] && Exponent[u,x]===1


(* If u(x) is equivalent to an expression of the form a+b*x+c*x^2, QuadraticQ[u,x] returns True;
	else it returns False. *)
QuadraticQ[u_,x_Symbol] :=
  PolynomialQ[u,x] && Exponent[u,x]===2


(* If u(x) is equivalent to an expression of the form x^n, MonomialQ[u,x] returns True;
	else it returns False.  Note that not all monomials are polynomials. *)
MonomialQ[u_,x_Symbol] :=
  u===x || PowerQ[u] && u[[1]]===x && FreeQ[u[[2]],x]


(* If u(x) is equivalent to an expression of the form a+b*x^n, BinomialQ[u,x] returns True;
	else it returns False.  Note that not all binomials are polynomials. *)
BinomialQ[u_,x_Symbol] :=
  NotFalseQ[BinomialTest[u,x]]


(* If u(x) is equivalent to an expression of the form a+b/x, InverseLinearQ[u,x] returns true;
	else it returns False. *)
InverseLinearQ[u_,x_Symbol] :=
  Module[{lst=BinomialTest[u,x]},
  If[FalseQ[lst],
    False,
  lst[[3]]===-1]]


(* If u(x) is a sum and each term is free of x or an expression of the form a*x^n,
	MonomialSumQ[u,x] returns True; else it returns False. *)
MonomialSumQ[u_,x_Symbol] :=
  SumQ[u] && Catch[
	Scan[Function[If[FreeQ[#,x] || MonomialQ[DependentFactors[#,x],x], Null, Throw[False]]],u];
    True]


(* ::Subsubsection::Closed:: *)
(*Polynomial Terms Functions*)


(* If u(x) is an expression of the form a*x^n where n is zero or a positive integer,
	PolynomialTermQ[u,x] returns True; else it returns False. *)
PolynomialTermQ[u_,x_Symbol] :=
  FreeQ[u,x] || MatchQ[u,a_.*x^n_. /; FreeQ[a,x] && IntegerQ[n] && n>0]


(* u(x) is a sum.  PolynomialTerms[u,x] returns the sum of the polynomial terms of u(x). *)
PolynomialTerms[u_,x_Symbol] :=
  Map[Function[If[PolynomialTermQ[#,x],#,0]],u]


(* u(x) is a sum.  NonpolynomialTerms[u,x] returns the sum of the nonpolynomial terms of u(x). *)
NonpolynomialTerms[u_,x_Symbol] :=
  Map[Function[If[PolynomialTermQ[#,x],0,#]],u]


(* ::Subsubsection::Closed:: *)
(*Binomial Test Function*)


(* If u(x) is equivalent to an expression of the form a+b*x^n where a,b and n are free of x,
	BinomialTest[u,x] returns the list {a,b,n}; else it returns False.
	Note that not all binomials are polynomials. *)
BinomialTest[u_,x_Symbol] :=
  If[u===x,
    {0,1,1},
  If[FreeQ[u,x],
    {0,u,0},
  If[PowerQ[u],
    If[u[[1]]===x && FreeQ[u[[2]],x],
      {0,1,u[[2]]},
    False],
  Module[{lst1,lst2},
  If[ProductQ[u],
    lst1=BinomialTest[First[u],x];
    If[FalseQ[lst1],
      False,
    lst2=BinomialTest[Rest[u],x];
    If[FalseQ[lst2],
      False,
    Module[{a,b,c,d,m,n},
    {a,b,m}=lst1;
    {c,d,n}=lst2;
    If[m===0,
      {b*c,b*d,n},
    If[n===0,
      {a*d,b*d,m},
    If[a===0,
      If[c===0,
        {0,b*d,m+n},
      If[m+n===0,
        {b*d,b*c,m},
      False]],
    If[c===0,
      If[m+n===0,
        {b*d,a*d,n},
      False],
    False]]]]]]],  
  If[SumQ[u],
    lst1=BinomialTest[First[u],x];
    If[FalseQ[lst1],
      False,
    lst2=BinomialTest[Rest[u],x];
    If[FalseQ[lst2],
      False,
    Module[{a,b,c,d,m,n},
    {a,b,m}=lst1;
    {c,d,n}=lst2;
    If[m===0,
      {b+c,d,n},
    If[n===0,
      {a+d,b,m},
    If[m===n,
      {a+c,b+d,m},
    False]]]]]],
  False]]]]]]


(* ::Subsubsection::Closed:: *)
(*Trinomial Test Function*)


(* If u(x) is equivalent to an expression of the form a+b*x^n+c*x^(2*n) where n>0 is an integer
	and a, b and c are nonzero, TrinomialTest[u,x] returns the list {a,b,c,n}; else it
	returns False. *)
TrinomialTest[u_,x_Symbol] :=
  If[PolynomialQ[u,x],
    Module[{lst=Exponent[u,x,List]},
    If[Length[lst]==3 && 2*lst[[2]]===lst[[3]] && lst[[1]]===0,
      {Coefficient[u,x,0],Coefficient[u,x,lst[[2]]],Coefficient[u,x,lst[[3]]],lst[[2]]},
    False]],
  False]


(* ::Subsubsection::Closed:: *)
(*Linear Power Test Function*)


(* If u(x) is equivalent to an expression of the form (a+b*x)^n where n>1 is an integer,
	LinearPowerTest[u,x] returns u(x) in the form (a+b*x)^n; else it returns False. *)
LinearPowerTest[u_,x_Symbol] :=
  If[PolynomialQ[u,x],
    Module[{v=FactorSquareFree[u]},
    If[PowerQ[v] && LinearQ[v[[1]],x],
      v,
    False]],
  False]


(* ::Subsubsection::Closed:: *)
(*Perfect Power Test Function*)


(* If u(x) is equivalent to a polynomial raised to an integer power greater than 1,
	PerfectPowerTest[u,x] returns u(x) as an expanded polynomial raised to the power;
	else it returns False. *)
PerfectPowerTest[u_,x_Symbol] :=
  If[PolynomialQ[u,x],
    Module[{lst=FactorSquareFreeList[u],gcd=0,v=1},
    If[lst[[1]]==={1,1},
      lst=Rest[lst]];
    Scan[Function[gcd=GCD[gcd,#[[2]]]],lst];
    If[gcd>1,
      Scan[Function[v=v*#[[1]]^(#[[2]]/gcd)],lst];
      Expand[v]^gcd,
    False]],
  False]


(* ::Subsubsection::Closed:: *)
(*Square Free Factor Test Function*)


(* If u(x) can be square free factored, SquareFreeFactorTest[u,x] returns u(x) in
	factored form; else it returns False. *)
(* SquareFreeFactorTest[u_,x_Symbol] :=
  If[PolynomialQ[u,x],
    Module[{v=FactorSquareFree[u]},
    If[PowerQ[v] || ProductQ[v],
      v,
    False]],
  False] *)


(* ::Section::Closed:: *)
(*Rational Function Functions*)


(* ::Subsubsection::Closed:: *)
(*Rational Function Recognizer*)


(* If u(x) is a polynomial or rational function, RationalFunctionQ[u,x] returns True; else
	it returns False. *)
RationalFunctionQ[u_,x_Symbol] :=
  If[AtomQ[u] || FreeQ[u,x],
    True,
  If[IntegerPowerQ[u],
    RationalFunctionQ[u[[1]],x],
  If[ProductQ[u] || SumQ[u],
    Catch[Scan[Function[If[RationalFunctionQ[#,x],Null,Throw[False]]],u];True],
  False]]]


(* ::Subsubsection::Closed:: *)
(*Linear Quotient Test Function*)


(* If u is equivalent to an expression of the form (a+b*x)/(c+d*x), LinearQuotientTest[u,x]
	returns {a,b,c,d}; else it returns False. *)
LinearQuotientTest[u_,x_Symbol] :=
  If[RationalFunctionQ[u,x],
    Module[{v=Together[u],numr,denr},
    If[LinearQ[numr=Numerator[v],x] && LinearQ[denr=Denominator[v],x],
      Join[CoefficientList[numr,x],CoefficientList[denr,x]],
    False]],
  False]


(* ::Subsubsection::Closed:: *)
(*Rational Function Terms Functions*)


(* u(x) is a sum.  RationalFunctionTerms[u,x] returns the sum of the nonpolynomial rational
	function terms of u(x). *)
RationalFunctionTerms[u_,x_Symbol] :=
  Map[Function[If[RationalFunctionQ[#,x] && Not[PolynomialQ[#,x]],#,0]],u]


(* u(x) is a sum.  NonrationalFunctionTerms[u,x] returns the sum of the polynomial and
	nonrational function terms of u(x). *)
NonrationalFunctionTerms[u_,x_Symbol] :=
  Map[Function[If[RationalFunctionQ[#,x] && Not[PolynomialQ[#,x]],0,#]],u]


(* ::Section::Closed:: *)
(*Inverse Function Free Recognizer*)


(* If u is free of inverse functions or calculus functions involving x,
	InverseFunctionFreeQ[u,x] returns true; else it returns False. *)
InverseFunctionFreeQ[u_,x_Symbol] :=
  If[AtomQ[u],
    True,
  If[LogQ[u] || InverseTrigQ[u] || InverseHyperbolicQ[u] || CalculusQ[u],
    If[FreeQ[u,x],
      True,
    False],
  Catch[Scan[Function[If[InverseFunctionFreeQ[#,x],Null,Throw[False]]],u];True]]]


(* ::Section::Closed:: *)
(*Normalization Functions*)


(* NormalizeExpression[u,x] returns u(x) in normal form. *)
NormalizeExpression[u_,x_Symbol] :=
  Module[{v=Simplify[u]},
  If[v===u,
    NormalizeExpression[v,True,x],
  If[SumQ[v],
    v,
  If[PowerQ[v],
    If[MatchQ[v,(a_.+b_.*x^n_.)^p_ /; FreeQ[{a,b,n,p},x]],
	  v,
    If[MatchQ[v,(a_.+b_.*x+c_.*x^2)^n_ /; FreeQ[{a,b,c,n},x]],
      v,
    NormalizeExpression[v,True,x]]],
  If[ProductQ[v],
    If[MatchQ[v,x^m_.*(a_.+b_.*x^n_.)^p_. /; FreeQ[{a,b,n,p},x]],
	  v,
    If[MatchQ[v,(d_.+x^m_.)*(a_.+b_.*x+c_.*x^2)^n_. /; FreeQ[{a,b,c,d,n},x]],
      v,
(*  If[SumQ[First[v]] && NonsumQ[Rest[v]],
      Map[Function[Rest[v]*#],First[v]],
    If[SumQ[Rest[v]] && NonsumQ[First[v]],
      Map[Function[First[v]*#],Rest[v]], *)
    NormalizeExpression[v,True,x]]],
  NormalizeExpression[v,True,x]]]]]]

NormalizeExpression[u_,flag_,x_] :=
  Module[{tmp},
  If[IntegerPowerQ[u] && (tmp=MonomialFactor[u,x])[[1]]=!=0,
    x^tmp[[1]]*NormalizeExpression[tmp[[2]],False,x],
  If[NotFalseQ[tmp=BinomialTest[u,x]],
    tmp[[1]] + tmp[[2]]*x^tmp[[3]],
  If[NotFalseQ[tmp=LinearPowerTest[u,x]],
    tmp,
  If[NotFalseQ[tmp=LinearQuotientTest[u,x]],
    (tmp[[1]] + tmp[[2]]*x)/(tmp[[3]]+tmp[[4]]*x),
  If[NotFalseQ[tmp=TrinomialTest[u,x]],
    (* If[ZeroQ[tmp[[2]]^2-4*tmp[[1]]*tmp[[3]]],
      (tmp[[2]]+2*tmp[[3]]*x^tmp[[4]])^2/(4*tmp[[3]]), *)
    tmp[[1]] + tmp[[2]]*x^tmp[[4]] + tmp[[3]]*x^(2*tmp[[4]]),
  If[flag &&
		(ProductQ[u] || IntegerPowerQ[u]) &&
		PolynomialQ[tmp=Denominator[u],x] &&
		(Exponent[tmp,x]<=3 || Not[BinomialQ[tmp,x]]) &&
		SumQ[tmp=Apart[u,x]],
    tmp,
  If[flag && SumQ[tmp=Expand[u,x]],
    tmp,
  If[PowerQ[u],
    If[QuadraticQ[u[[1]],x] && Not[BinomialQ[u[[1]],x]],
      Apart[u[[1]],x]^NormalizeExpression[u[[2]],False,x],
    tmp=MonomialFactor[Together[u[[1]]],x];
    If[tmp[[1]]===0,
      tmp=tmp[[2]],
    tmp=x^tmp[[1]]*Apart[tmp[[2]],x]];
    NormalizeExpression[tmp,False,x]^NormalizeExpression[u[[2]],False,x]],
  Map[Function[NormalizeExpression[#,False,x]],u]]]]]]]]]]


(* SimplifyCleanup[u_] :=
  If[AtomQ[u],
    u,
  If[IntegerPowerQ[u] && SumQ[u[[1]]] && RationalQ[First[u[[1]]]] && First[u[[1]]]<0,
    (-1)^u[[2]]*(-First[u[[1]]]-SimplifyCleanup[Rest[u[[1]]]])^u[[2]],
  Map[Function[SimplifyCleanup[#]],u]]] *)


(* ::Section::Closed:: *)
(*Common Factors*)


CommonFactors[u_,v_] :=
  CommonFactors[{u,v}]

(* lst is a n-element list of terms.  CommonFactors[lst] returns a n+1-element list whose first
	element is the product of the factors common to all terms of lst, and whose remaining
	elements are quotients of each term divided by the common factor. *)
CommonFactors [lst_List] :=
  Module[{lst1=lst,common=1,lst2=Table[1,{Length[lst]}],lst3,lst4,base,num},
  While[True,
    lst3=Map[LeadFactor,lst1];
    ( If[Apply[SameQ,lst3],
        common=common*lst3[[1]];
        lst1=Map[RemainingFactors,lst1],
      If[MapAnd[RationalQ,lst3],
        num=Apply[GCD,lst3];
        ( If[MapAnd[Function[#<0],lst3],
            num=-num] );
	    common=common*num;
        lst2=Map2[Function[#1*#2/num],lst2,lst3];
        lst1=Map[RemainingFactors,lst1],
      If[MapAnd[Function[LogQ[#] && PositiveIntegerQ[First[#]]],lst3] &&
           MapAnd[RationalQ,lst4=Map[Function[FullSimplify[#/First[lst3]]],lst3]],
        num=Apply[GCD,lst4];
        common=common*Log[(First[lst3][[1]])^num];
        lst2=Map2[Function[#1*#2/num],lst2,lst4];
        lst1=Map[RemainingFactors,lst1],
      If[Apply[SameQ,Map[LeadBase,lst1]] && MapAnd[RationalQ,lst4=Map[LeadDegree,lst1]],
        num=Smallest[lst4];
        ( If[num!=0,
            base=LeadBase[lst1[[1]]];
            common=common*base^num;
            lst2=Map2[Function[#1*base^(#2-num)],lst2,lst4]] );
        lst1=Map[RemainingFactors,lst1],
      num=MostMainFactorPosition[lst3];
      lst2=ReplacePart[lst2,lst3[[num]]*lst2[[num]],num];      
      lst1=ReplacePart[lst1,RemainingFactors[lst1[[num]]],num]]]]] );
    If[MapAnd[Function[#===1],lst1],
      Return[Prepend[lst2,common]]]]]


MostMainFactorPosition[lst_List] :=
  Module[{factor=1,num=1},
  Do[If[FactorOrder[lst[[i]],factor]>0,factor=lst[[i]];num=i],{i,Length[lst]}];
  num]


FactorOrder[u_,v_] :=
  If[u===1,
    If[v===1,
      0,
    -1],
  If[v===1,
    1,
  Order[u,v]]]


Smallest[num1_,num2_] :=
  If[num1>0,
    If[num2>0,
      Min[num1,num2],
    0],
  If[num2>0,
    0,
  Max[num1,num2]]]

Smallest[lst_List] :=
  Module[{num=lst[[1]]},
  Scan[Function[num=Smallest[num,#]],Rest[lst]];
  num]


(* ::Section::Closed:: *)
(*Content Factoring*)


(* ::Subsection::Closed:: *)
(*Monomial factoring*)


(* MonomialFactor[u,x] returns the list {n,v(x)} where u(x) is equivalent to x^n*v(x). *) 
MonomialFactor[u_,x_Symbol] :=
  If[AtomQ[u],
    If[u===x,
      {1,1},
    {0,u}],
  If[PowerQ[u],
    If[IntegerQ[u[[2]]],
      Module[{lst=MonomialFactor[u[[1]],x]},
      {lst[[1]]*u[[2]],lst[[2]]^u[[2]]}],
    If[u[[1]]===x,
      {u[[2]],1},
    {0,u}]],
  If[ProductQ[u],
    Module[{lst1=MonomialFactor[First[u],x],lst2=MonomialFactor[Rest[u],x]},
    {lst1[[1]]+lst2[[1]],lst1[[2]]*lst2[[2]]}],
  If[SumQ[u],
    Module[{lst1=MonomialFactor[First[u],x],lst2=MonomialFactor[Rest[u],x],deg},
    If[RationalQ[lst1[[1]]],
      If[RationalQ[lst2[[1]]],
        deg=Min[lst1[[1]],lst2[[1]]],
      deg=lst1[[1]]],
    If[RationalQ[lst2[[1]]],
      deg=lst2[[1]],
    deg=Simplify[lst1[[1]]-lst2[[1]]];
    If[RationalQ[deg],
      If[deg>0,
        deg=lst2[[1]],
      deg=lst1[[1]]],
    deg=0]]];
    If[deg===0,
      {0,u},
    {deg,x^(lst1[[1]]-deg)*lst1[[2]]+x^(lst2[[1]]-deg)*lst2[[2]]}]],    
  {0,u}]]]]


(* ::Subsection::Closed:: *)
(*Constant factoring*)


(* ConstantFactor[u,x] returns a 2-element list of the factors of u(x) free of x and the 
	factors not free of u(x).  Common constant factors of the terms of sums are also collected.
	Compare with the more passive functions IndependentFactors and DependentFactors. *)
ConstantFactor[u_,x_Symbol] :=
  If[AtomQ[u],
    If[u===x,
      {1,u},
    {u,1}],
  If[IntegerPowerQ[u],
    Module[{lst=ConstantFactor[u[[1]],x]},
    {lst[[1]]^u[[2]],lst[[2]]^u[[2]]}],
  If[ProductQ[u],
    Module[{lst=Map[Function[ConstantFactor[#,x]],Apply[List,u]]},
    {Apply[Times,Map[First,lst]],Apply[Times,Map[Second,lst]]}],
  If[SumQ[u],
    Module[{lst1=Map[Function[ConstantFactor[#,x]],Apply[List,u]]},
    If[Apply[SameQ,Map[Second,lst1]],
      {Apply[Plus,Map[First,lst1]],lst1[[1,2]]},
    Module[{lst2=CommonFactors[Map[First,lst1]]},
    {First[lst2],Apply[Plus,Map2[Times,Rest[lst2],Map[Second,lst1]]]}]]],
  If[FreeQ[u,x],
    {u,1},
  {1,u}]]]]]


(* ::Section::Closed:: *)
(*Function Of Functions*)


(* ::Subsection::Closed:: *)
(*Function of linear*)


(* If u(x) is equivalent to an expression of the form f(a+b*x) and not the case that a=0 and b=1,
	FunctionOfLinear[u,x] returns the list {f(x),a,b}; else it returns False. *)
FunctionOfLinear[u_,x_Symbol] :=
  Module[{lst=FunctionOfLinear[u,False,False,x]},
  If[FalseQ[lst] || FalseQ[lst[[1]]],
    False,
  {FunctionOfLinearSubst[u,lst[[1]],lst[[2]],x],lst[[1]],lst[[2]]}]]


FunctionOfLinear[u_,a_,b_,x_] :=
  If[u===x,
    False,
  If[AtomQ[u],
    {a,b},
  If[CalculusQ[u],
    False,
  If[LinearQ[u,x],
    If[FalseQ[a],
      {Coefficient[u,x,0],Coefficient[u,x,1]},
    If[ZeroQ[a],
      If[ZeroQ[Coefficient[u,x,0]],
        Module[{v=CommonFactors[Coefficient[u,x,1],b][[1]]},
        If[v===1,
          False,
        {0,v}]],
      FunctionOfLinear[Coefficient[u,x,1]*x,a,b,x]],
    If[Coefficient[u,x,0]/a===Coefficient[u,x,1]/b,
      Module[{lst=CommonFactors[Coefficient[u,x,1],b]},
      {a/lst[[3]],lst[[1]]}],
(*  FunctionOfLinear[u,0,b,x]]]], *)
    False]]],
  If[PowerQ[u] && FreeQ[u[[1]],x],
    FunctionOfLinear[Log[u[[1]]]*u[[2]],a,b,x],
  Module[{lst},
  If[ProductQ[u] && NonzeroQ[(lst=MonomialFactor[u,x])[[1]]],
    If[IntegerQ[lst[[1]]] && lst[[1]]!=-1 && FreeQ[lst[[2]],x],
      If[RationalQ[LeadFactor[lst[[2]]]] && LeadFactor[lst[[2]]]<0,
        FunctionOfLinear[DivideDegreesOfFactors[-lst[[2]],lst[[1]]]*x,a,b,x],
      FunctionOfLinear[DivideDegreesOfFactors[lst[[2]],lst[[1]]]*x,a,b,x]],
    False],
  lst={a,b};
  Catch[
  Scan[Function[lst=FunctionOfLinear[#,lst[[1]],lst[[2]],x];If[lst===False,Throw[False]]],u];
  lst]]]]]]]]


FunctionOfLinearSubst[u_,a_,b_,x_] :=
  If[AtomQ[u],
    u,
  If[LinearQ[u,x],
    If[ZeroQ[a],
      Coefficient[u,x,0]+Coefficient[u,x,1]/b*x,
    Coefficient[u,x,1]/b*x],
  If[PowerQ[u] && FreeQ[u[[1]],x],
    E^FullSimplify[FunctionOfLinearSubst[Log[u[[1]]]*u[[2]],a,b,x]],
  Module[{lst},
  If[ProductQ[u] && NonzeroQ[(lst=MonomialFactor[u,x])[[1]]],
    If[RationalQ[LeadFactor[lst[[2]]]] && LeadFactor[lst[[2]]]<0,
      -FunctionOfLinearSubst[DivideDegreesOfFactors[-lst[[2]],lst[[1]]]*x,a,b,x]^lst[[1]],
    FunctionOfLinearSubst[DivideDegreesOfFactors[lst[[2]],lst[[1]]]*x,a,b,x]^lst[[1]]],
  Map[Function[FunctionOfLinearSubst[#,a,b,x]],u]]]]]]


(* DivideDegreesOfFactors[u,n] returns the product of the base of the factors of u raised to
	the degree of the factors divided by n. *)
DivideDegreesOfFactors[u_,n_] :=
  If[ProductQ[u],
    Map[Function[LeadBase[#]^(LeadDegree[#]/n)],u],
  LeadBase[u]^(LeadDegree[u]/n)]


(* ::Subsection::Closed:: *)
(*Function of expression predicate*)


(* If u(x) is a function of the form f(v) where f is independent of x,
	FunctionOfQ[v,u,x] returns True; else it returns False. *)
FunctionOfQ[v_,u_,x_Symbol] :=
  If[AtomQ[v],
    True,
  If[PowerQ[v] && RationalQ[v[[2]]] && v[[2]]!=-1,
    FunctionOfPowerQ[u,v[[1]],v[[2]],x],
  If[ExpQ[v],
    FunctionOfExpQ[u,v[[2]],x],
  If[(SinQ[v] || CscQ[v]),
    FunctionOfSinQ[u,v[[1]],x],
  If[(CosQ[v] || SecQ[v]),
    FunctionOfCosQ[u,v[[1]],x],
  If[(TanQ[v] || CotQ[v]),
    FunctionOfTanQ[u,v[[1]],x],
  If[(SinhQ[v] || CschQ[v]),
    FunctionOfSinhQ[u,v[[1]],x],
  If[(CoshQ[v] || SechQ[v]),
    FunctionOfCoshQ[u,v[[1]],x],
  If[(TanhQ[v] || CothQ[v]),
    FunctionOfTanhQ[u,v[[1]],x],
  FunctionOfExpnQ[u,v,x]]]]]]]]]]


(* ::Subsubsection::Closed:: *)
(*Function of Expression Predicates*)


FunctionOfExpnQ[u_,v_,x_] :=
  If[u===v,
    True,
  If[AtomQ[u],
    u=!=x,
  If[CalculusQ[u],
    False,
  Catch[Scan[Function[If[FunctionOfExpnQ[#,v,x],Null,Throw[False]]],u];True]]]]


FunctionOfPowerQ[u_,bas_,deg_,x_] :=
  If[AtomQ[u],
    u=!=x,
  If[CalculusQ[u],
    False,
  If[PowerQ[u] && u[[1]]===bas && RationalQ[u[[2]]] && IntegerQ[u[[2]]/deg],
    True,
  Catch[Scan[Function[If[FunctionOfPowerQ[#,bas,deg,x],Null,Throw[False]]],u];True]]]]


(* ::Subsubsection::Closed:: *)
(*Function of Exponential Function Predicate*)


(* If u(x) is a function of the form f(E^v) where f is independent of x,
	FunctionOfExpQ[u,v,x] returns True; else it returns False. *)
FunctionOfExpQ[u_,v_,x_] :=
  If[AtomQ[u],
    u=!=x,
  If[CalculusQ[u],
    False,
  If[ExpQ[u] && IntegerQ[u[[2]]/v],
    True,
  If[PowerQ[u] && FreeQ[u[[1]],x] && SumQ[u[[2]]],
    FunctionOfExpQ[u[[1]]^First[u[[2]]],v,x] && FunctionOfExpQ[u[[1]]^Rest[u[[2]]],v,x],
  Catch[Scan[Function[If[FunctionOfExpQ[#,v,x],Null,Throw[False]]],u];True]]]]]


(* ::Subsubsection::Closed:: *)
(*Function of Trig Function Predicates*)


(* If u(x) is a function of the form f(Sin[v],Csc[v],Cos[v/2]*Sin[v/2]) where f is independent
	of x, FunctionOfSinQ[u,v,x] returns True; else it returns False. *)
FunctionOfSinQ[u_,v_,x_] :=
  If[AtomQ[u],
    u=!=x,
  If[CalculusQ[u],
    False,
  If[(SinQ[u] || CscQ[u]) && u[[1]]===v,
    True,
  If[(CosQ[u] || SecQ[u]) && u[[1]]===2*v,
    True,
  If[PowerQ[u] && EvenQ[u[[2]]] && TrigQ[u[[1]]] && u[[1,1]]===v,
    True,
  If[ProductQ[u] && CosQ[u[[1]]] && SinQ[u[[2]]] && u[[1,1]]===u[[2,1]]===v/2,
    FunctionOfSinQ[Drop[u,2]],
  Catch[Scan[Function[If[FunctionOfSinQ[#,v,x],Null,Throw[False]]],u];True]]]]]]]


(* If u(x) is a function of the form f(Cos[v],Sec[v]) where f is independent of x,
	FunctionOfCosQ[u,v,x] returns True; else it returns False. *)
FunctionOfCosQ[u_,v_,x_] :=
  If[AtomQ[u],
    u=!=x,
  If[CalculusQ[u],
    False,
  If[(CosQ[u] || SecQ[u]) && (u[[1]]===v || u[[1]]===2*v),
    True,
  If[PowerQ[u] && EvenQ[u[[2]]] && TrigQ[u[[1]]] && u[[1,1]]===v,
    True,
  Catch[Scan[Function[If[FunctionOfCosQ[#,v,x],Null,Throw[False]]],u];True]]]]]


(* If u(x) is a function of the form f(Tan[v],Cot[v]) where f is independent of x,
	FunctionOfTanQ[u,v,x] returns True; else it returns False. *)
FunctionOfTanQ[u_,v_,x_] :=
  If[AtomQ[u],
    u=!=x,
  If[CalculusQ[u],
    False,
  If[(TanQ[u] || CotQ[u]) && u[[1]]===v,
    True,
  If[PowerQ[u] && EvenQ[u[[2]]] && TrigQ[u[[1]]] && u[[1,1]]===v,
    True,
  Catch[Scan[Function[If[FunctionOfTanQ[#,v,x],Null,Throw[False]]],u];True]]]]]


(* ::Subsubsection::Closed:: *)
(*Function of Hyperbolic Function Predicates*)


(* If u(x) is a function of the form f(Sinh[v],Csch[v],Cosh[v/2]*Sinh[v/2]) where f is
	independent of x, FunctionOfSinhQ[u,v,x] returns True; else it returns False. *)
FunctionOfSinhQ[u_,v_,x_] :=
  If[AtomQ[u],
    u=!=x,
  If[CalculusQ[u],
    False,
  If[(SinhQ[u] || CschQ[u]) && u[[1]]===v,
    True,
  If[(CoshQ[u] || SechQ[u]) && u[[1]]===2*v,
    True,
  If[PowerQ[u] && EvenQ[u[[2]]] && HyperbolicQ[u[[1]]] && u[[1,1]]===v,
    True,
  If[ProductQ[u] && CoshQ[u[[1]]] && SinhQ[u[[2]]] && u[[1,1]]===u[[2,1]]===v/2,
    FunctionOfSinhQ[Drop[u,2]],
  Catch[Scan[Function[If[FunctionOfSinhQ[#,v,x],Null,Throw[False]]],u];True]]]]]]]


(* If u(x) is a function of the form f(Cosh[v],Sech[v]) where f is independent of x,
	FunctionOfCoshQ[u,v,x] returns True; else it returns False. *)
FunctionOfCoshQ[u_,v_,x_] :=
  If[AtomQ[u],
    u=!=x,
  If[CalculusQ[u],
    False,
  If[(CoshQ[u] || SechQ[u]) && (u[[1]]===v || u[[1]]===2*v),
    True,
  If[PowerQ[u] && EvenQ[u[[2]]] && HyperbolicQ[u[[1]]] && u[[1,1]]===v,
    True,
  Catch[Scan[Function[If[FunctionOfCoshQ[#,v,x],Null,Throw[False]]],u];True]]]]]


(* If u(x) is a function of the form f(Tanh[v],Coth[v]) where f is independent of x,
	FunctionOfTanhQ[u,v,x] returns True; else it returns False. *)
FunctionOfTanhQ[u_,v_,x_] :=
  If[AtomQ[u],
    u=!=x,
  If[CalculusQ[u],
    False,
  If[(TanhQ[u] || CothQ[u]) && u[[1]]===v,
    True,
  If[PowerQ[u] && EvenQ[u[[2]]] && HyperbolicQ[u[[1]]] && u[[1,1]]===v,
    True,
  Catch[Scan[Function[If[FunctionOfTanhQ[#,v,x],Null,Throw[False]]],u];True]]]]]


(* ::Subsection::Closed:: *)
(*Function of trig functions predicate*)


(* If u(x) is equivalent to an expression of the form f(Sin[x],Cos[x],Tan[x],Cot[x],Sec[x],Csc[x])
	where f is independent of x, FunctionOfTrigQ[u,x] returns True; else it returns False. *)
FunctionOfTrigQ[u_,x_Symbol] :=
  If[AtomQ[u],
    u=!=x,
  If[CalculusQ[u],
    False,
  If[TrigQ[u] && IntegerQ[u[[1]]/x],
    True,
  Catch[Scan[Function[If[FunctionOfTrigQ[#,x],Null,Throw[False]]],u];True]]]]


(* ::Subsection::Closed:: *)
(*Function of hyperbolic functions predicate*)


(* If u(x) is equivalent to a function of the form f(Sinh[x],Cosh[x],Tanh[x],Coth[x],Sech[x],Csch[x])
	where f is independent of x, FunctionOfHyperbolicQ[u,x] returns True; else it returns False. *)
FunctionOfHyperbolicQ[u_,x_Symbol] :=
  If[AtomQ[u],
    u=!=x,
  If[CalculusQ[u],
    False,
  If[HyperbolicQ[u] && IntegerQ[u[[1]]/x],
    True,
  Catch[Scan[Function[If[FunctionOfHyperbolicQ[#,x],Null,Throw[False]]],u];True]]]]


(* ::Subsection::Closed:: *)
(*Function of dense polynomials*)


(* If all occurrences of x in u(x) are in dense polynomials, FunctionOfDensePolynomialsQ[u,x]
	returns True; else it returns False. *)
FunctionOfDensePolynomialsQ[u_,x_Symbol] :=
  If[FreeQ[u,x],
    True,
  If[PolynomialQ[u,x],
    Length[Exponent[u,x,List]]>1,
  Catch[
  Scan[Function[If[FunctionOfDensePolynomialsQ[#,x],Null,Throw[False]]],u];
  True]]]


(* ::Subsection::Closed:: *)
(*Function of logarithm*)


(* If u(x) is equivalent to an expression of the form f(Log[a*x^n]), FunctionOfLog[u,x] returns
	the list {f(x),a*x^n,n}; else it returns False. *)
FunctionOfLog[u_,x_Symbol] :=
  Module[{lst=FunctionOfLog[u,False,False,x]},
  If[FalseQ[lst] || FalseQ[lst[[2]]],
    False,
  lst]]


FunctionOfLog[u_,v_,n_,x_] :=
  If[AtomQ[u],
    If[u===x,
      False,
    {u,v,n}],
  If[CalculusQ[u],
    False,
  Module[{lst},
  If[LogQ[u] && NotFalseQ[lst=BinomialTest[u[[1]],x]] && ZeroQ[lst[[1]]],
    If[FalseQ[v] || u[[1]]===v,
      {x,u[[1]],lst[[3]]},
    False],
  lst={0,v,n};
  Catch[
    {Map[Function[lst=FunctionOfLog[#,lst[[2]],lst[[3]],x];
				  If[lst===False,Throw[False],lst[[1]]]],
			u],lst[[2]],lst[[3]]}]]]]]


(* ::Subsection::Closed:: *)
(*Power of variable*)


(* If m is an integer, u is an expression of the form f(x^n), g=gcd(m,n), and g>1,
   PowerVariableExpn[u,m,x] returns the list {x^(m/g)*f(x^(n/g)),g}; else it returns False. *)
PowerVariableExpn[u_,m_,x_Symbol] :=
  Module[{deg=PowerVariableDegree[u,m,x]},
  If[FalseQ[deg],
    False,
  {x^(m/deg)*PowerVariableSubst[u,deg,x],deg}]]


PowerVariableDegree[u_,m_,x_Symbol] :=
  If[IndependentQ[u,x],
    m,
  If[AtomQ[u] || CalculusQ[u],
    False,
  If[PowerQ[u] && u[[1]]===x,
    If[IntegerQ[u[[2]]] && GCD[m,u[[2]]]>1,
      GCD[m,u[[2]]],
    False],
  Catch[Module[{deg=m},
  Scan[Function[deg=PowerVariableDegree[#,deg,x];If[deg===False,Throw[False]]],u];
  deg]]]]]


PowerVariableSubst[u_,m_,x_Symbol] :=
  If[IndependentQ[u,x] || AtomQ[u] ||CalculusQ[u],
    u,
  If[PowerQ[u] && u[[1]]===x,
    x^(u[[2]]/m),
  Map[Function[PowerVariableSubst[#,m,x]],u]]]


(* ::Subsection::Closed:: *)
(*Root of linear expression*)


(* If u(x) is an expression of the form f((a+b*x)^(1/n),x) where n>1 is an integer,
LinearRootExpn[u,x] returns the list {f(x,(x^n-a)/b),n,a+b*x,b}; else it returns False. *)
LinearRootExpn[u_,x_Symbol] :=
  Module[{lst=LinearRootExpn[u,1,False,x],a,b,n},
  If[FalseQ[lst] || FalseQ[lst[[2]]],
    False,
  a=Coefficient[lst[[2]],x,0];
  b=Coefficient[lst[[2]],x,1];
  n=lst[[1]];
  {Simplify[RootSubst[u,lst[[2]],n,-a/b+x^n/b,x]],n,lst[[2]],b}]]


LinearRootExpn[u_,n_,v_,x_] :=
  If[AtomQ[u] || IndependentQ[u,x],
    {n,v},
  If[CalculusQ[u],
    False,
  If[FractionalPowerQ[u] && LinearQ[u[[1]],x] && (FalseQ[v] || u[[1]]===v),
    {LCM[Denominator[u[[2]]],n],u[[1]]},
  Catch[Module[{lst={n,v}},
  Scan[Function[lst=LinearRootExpn[#,lst[[1]],lst[[2]],x];If[lst===False,Throw[False]]],u];
  lst]]]]]


(* ::Subsection::Closed:: *)
(*Root of inverse linear expression*)


(* If u(x) is an expression of the form f((a+b/x)^(1/n),x) where n>1 is an integer, 
	InverseLinearRootExpn[u,x] returns the list {f(x,b/(x^n-a))/(x^n-a)^2,n,a+b/x,b};
	else it returns False. *)
InverseLinearRootExpn[u_,x_Symbol] :=
  Module[{lst=InverseLinearRootExpn[u,1,False,x],a,b,n},
  If[FalseQ[lst] || FalseQ[lst[[2]]],
    False,
  n=BinomialTest[lst[[2]],x];
  a=n[[1]];
  b=n[[2]];
  n=lst[[1]];
  {Simplify[RootSubst[u,lst[[2]],n,b/(x^n-a),x]/(x^n-a)^2],n,lst[[2]],b}]]


InverseLinearRootExpn[u_,n_,v_,x_] :=
  If[AtomQ[u] || IndependentQ[u,x],
    {n,v},
  If[CalculusQ[u],
    False,
  If[FractionalPowerQ[u] && InverseLinearQ[u[[1]],x] && (FalseQ[v] || u[[1]]===v),
    {LCM[Denominator[u[[2]]],n],u[[1]]},
  Catch[Module[{lst={n,v}},
  Scan[Function[lst=InverseLinearRootExpn[#,lst[[1]],lst[[2]],x];If[lst===False,Throw[False]]],u];
  lst]]]]]


(* ::Subsection::Closed:: *)
(*Squareroot of quadratic expression*)


(*
Euler substitution #2:
  If u is an expression of the form f(Sqrt[a+b*x+c*x^2],x), f(x,x) is a rational function, and
	PosQ[c], FunctionOfSquareRootOfQuadratic[u,x] returns the 3-element list {
		f((a*Sqrt[c]+b*x+Sqrt[c]*x^2)/(b+2*Sqrt[c]*x),(-a+x^2)/(b+2*Sqrt[c]*x))*
		  (a*Sqrt[c]+b*x+Sqrt[c]*x^2)/(b+2*Sqrt[c]*x)^2,
		Sqrt[c]*x+Sqrt[a+b*x+c*x^2], 2 };

Euler substitution #1:
  If u is an expression of the form f(Sqrt[a+b*x+c*x^2],x), f(x,x) is a rational function, and
	PosQ[a], FunctionOfSquareRootOfQuadratic[u,x] returns the two element list {
		f((c*Sqrt[a]-b*x+Sqrt[a]*x^2)/(c-x^2),(-b+2*Sqrt[a]*x)/(c-x^2))*
		  (c*Sqrt[a]-b*x+Sqrt[a]*x^2)/(c-x^2)^2,
		(-Sqrt[a]+Sqrt[a+b*x+c*x^2])/x, 1 };

Euler substitution #3:
  If u is an expression of the form f(Sqrt[a+b*x+c*x^2],x), f(x,x) is a rational function, and
	NegQ[a] and NegQ[c], FunctionOfSquareRootOfQuadratic[u,x] returns the two element list {
		-Sqrt[b^2-4*a*c]*
		f(-Sqrt[b^2-4*a*c]*x/(c-x^2),-(b*c+c*Sqrt[b^2-4*a*c]+(-b+Sqrt[b^2-4*a*c])*x^2)/(2*c*(c-x^2)))*
		  x/(c-x^2)^2,
		2*c*Sqrt[a+b*x+c*x^2]/(b-Sqrt[b^2-4*a*c]+2*c*x), 3 };

  else it returns False. *)

FunctionOfSquareRootOfQuadratic[u_,x_Symbol] :=
  Module[{tmp=FunctionOfSquareRootOfQuadratic[u,False,x]},
  If[FalseQ[tmp] || FalseQ[tmp[[1]]],
    False,
  tmp=tmp[[1]];
  Module[{a=Coefficient[tmp,x,0],b=Coefficient[tmp,x,1],c=Coefficient[tmp,x,2],sqrt,q,r},
  If[ZeroQ[a] && ZeroQ[b],
    False,
  If[PosQ[c],
    sqrt=Rt[c,2];
    q=a*sqrt+b*x+sqrt*x^2;
    r=b+2*sqrt*x;
    {Simplify[SquareRootOfQuadraticSubst[u,q/r,(-a+x^2)/r,x]*q/r^2],sqrt*x+Sqrt[tmp],2},
  If[PosQ[a],
    sqrt=Rt[a,2];
    q=c*sqrt-b*x+sqrt*x^2;
    r=c-x^2;
    {Simplify[SquareRootOfQuadraticSubst[u,q/r,(-b+2*sqrt*x)/r,x]*q/r^2],(-sqrt+Sqrt[tmp])/x,1},
  sqrt=Rt[b^2-4*a*c,2];
  r=c-x^2;
  {Simplify[-sqrt*SquareRootOfQuadraticSubst[u,-sqrt*x/r,-(b*c+c*sqrt+(-b+sqrt)*x^2)/(2*c*r),x]*x/r^2],
		2*c*Sqrt[tmp]/(b-sqrt+2*c*x),3}]]]]]]


FunctionOfSquareRootOfQuadratic[u_,v_,x_Symbol] :=
  If[AtomQ[u] || IndependentQ[u,x],
    {v},
  If[FractionalPowerQ[u] && Denominator[u[[2]]]===2,
    If[PolynomialQ[u[[1]],x] && Exponent[u[[1]],x]===2,
      If[(FalseQ[v] || u[[1]]===v),
        {u[[1]]},
      False],
    FunctionOfSquareRootOfQuadratic[u[[1]],v,x]],
  If[IntegerPowerQ[u] || ProductQ[u] || SumQ[u],
    Catch[Module[{lst={v}},
    Scan[Function[lst=FunctionOfSquareRootOfQuadratic[#,lst[[1]],x];If[lst===False,Throw[False]]],u];
    lst]],
  False]]]


(* SquareRootOfQuadraticSubst[u,vv,xx,x] returns u with fractional powers replaced by vv raised
	the power and x replaced by xx. *)
SquareRootOfQuadraticSubst[u_,vv_,xx_,x_Symbol] :=
  If[AtomQ[u] || IndependentQ[u,x],
    If[u===x,
      xx,
    u],
  If[FractionalPowerQ[u] && Denominator[u[[2]]]===2,
    If[PolynomialQ[u[[1]],x] && Exponent[u[[1]],x]===2,
      vv^Numerator[u[[2]]],
    SquareRootOfQuadraticSubst[u[[1]],vv,xx,x]^u[[2]]],
  Map[Function[SquareRootOfQuadraticSubst[#,vv,xx,x]],u]]]


(* ::Section::Closed:: *)
(*Substitution Functions*)


(* ::Subsection::Closed:: *)
(*Substitute for variable expression*)


(* Subst[u,v,w] returns u with all nondummy occurences of v replaced by w *)
Subst[u_,v_,w_] :=
  If[u===v,
    w,
  If[AtomQ[u],
    u,
  If[PowerQ[u],
    If[PowerQ[v] && u[[1]]===v[[1]] && SumQ[u[[2]]],
      Subst[u[[1]]^First[u[[2]]],v,w]*Subst[u[[1]]^Rest[u[[2]]],v,w],
    Subst[u[[1]],v,w]^Subst[u[[2]],v,w]],
  If[SubstQ[u] && (u[[2]]===v || IndependentQ[u[[1]],v]),
    Subst[u[[1]],u[[2]],Subst[u[[3]],v,w]],
  Map[Function[Subst[#,v,w]],u]]]]] /;
AtomQ[u] || SubstQ[u] && (u[[2]]===v || IndependentQ[u[[1]],v]) ||
	Not[CalculusQ[u] && Not[FreeQ[v,u[[2]]]] || MemberQ[{Pattern,Defer,Hold,HoldForm},Head[u]]]


(* ::Subsection::Closed:: *)
(*Substitute for subexpressions*)


(* u is a function v.  SubstFor[v,u,w] returns f(w). *)
SubstFor[v_,u_,w_] :=
  If[AtomQ[v],
    Subst[u,v,w],
  If[PowerQ[v] && RationalQ[v[[2]]] && v[[2]]!=-1,
    SubstForPower[u,v[[1]],v[[2]],w],
  If[ExpQ[v],
    SubstForExp[u,v[[2]],w],
  If[SinQ[v],
    SubstForSin[u,v[[1]],w],
  If[CosQ[v],
    SubstForCos[u,v[[1]],w],
  If[TanQ[v],
    SubstForTan[u,v[[1]],w],
  If[SinhQ[v],
    SubstForSinh[u,v[[1]],w],
  If[CoshQ[v],
    SubstForCosh[u,v[[1]],w],
  If[TanhQ[v],
    SubstForTanh[u,v[[1]],w],
  SubstForExpn[u,v,w]]]]]]]]]]


(* ::Subsubsection::Closed:: *)
(*Substitution For Expression*)


SubstForExpn[u_,v_,w_] :=
  If[u===v,
    w,
  If[AtomQ[u],
    u,
  Map[Function[SubstForExpn[#,v,w]],u]]]


SubstForPower[u_,bas_,deg_,x_] :=
  If[AtomQ[u],
    u,
  If[PowerQ[u] && u[[1]]===bas && RationalQ[u[[2]]] && IntegerQ[u[[2]]/deg],
    x^(u[[2]]/deg),
  Map[Function[SubstForPower[#,bas,deg,x]],u]]]


(* ::Subsubsection::Closed:: *)
(*Substitution For Exponential Function*)


(* u is a function of the form f(E^v).  SubstForExp[u,v,x] returns f(x). *)
SubstForExp[u_,v_,x_] :=
  If[AtomQ[u],
    u,
  If[ExpQ[u] && IntegerQ[u[[2]]/v],
    x^(u[[2]]/v),
  If[PowerQ[u] && FreeQ[u[[1]],x] && SumQ[u[[2]]],
    SubstForExp[u[[1]]^First[u[[2]]],v,x]*SubstForExp[u[[1]]^Rest[u[[2]]],v,x],
  Map[Function[SubstForExp[#,v,x]],u]]]]


(* ::Subsubsection::Closed:: *)
(*Substitution For Trigonometric Functions*)


(* u is a function of the form f(Sin[v],Csc[v],Cos[v/2]*Sin[v/2]).
	SubstForSin[u,v,x] returns f(x,1/x,x/2). *)
SubstForSin[u_,v_,x_] :=
  If[AtomQ[u],
    u,
  If[SinQ[u] && u[[1]]===v,
    x,
  If[CscQ[u] && u[[1]]===v,
    1/x,
  If[CosQ[u] && u[[1]]===2*v,
    1-2*x^2,
  If[SecQ[u] && u[[1]]===2*v,
    1/(1-2*x^2),
  If[PowerQ[u] && EvenQ[u[[2]]] && TrigQ[u[[1]]] && u[[1,1]]===v,
    If[SinQ[u[[1]]],
      x^u[[2]],
    If[CosQ[u[[1]]],
      (1-x^2)^(u[[2]]/2),
    If[TanQ[u[[1]]],
      x^u[[2]]/(1-x^2)^(u[[2]]/2),
    If[CotQ[u[[1]]],
      (1-x^2)^(u[[2]]/2)/x^u[[2]],
    If[SecQ[u[[1]]],
      1/(1-x^2)^(u[[2]]/2),
    1/x^u[[2]]]]]]],
  If[ProductQ[u] && CosQ[u[[1]]] && SinQ[u[[2]]] && u[[1,1]]===u[[2,1]]===v/2,
    x/2*SubstForSin[Drop[u,2],v,x],
  Map[Function[SubstForSin[#,v,x]],u]]]]]]]]


(* u is a function of the form f(Cos[v],Sec[v]).  SubstForCos[u,v,x] returns f(x,1/x). *)
SubstForCos[u_,v_,x_] :=
  If[AtomQ[u],
    u,
  If[CosQ[u] && u[[1]]===v,
    x,
  If[SecQ[u] && u[[1]]===v,
    1/x,
  If[CosQ[u] && u[[1]]===2*v,
    -1+2*x^2,
  If[SecQ[u] && u[[1]]===2*v,
    -1/(1-2*x^2),
  If[PowerQ[u] && EvenQ[u[[2]]] && TrigQ[u[[1]]] && u[[1,1]]===v,
    If[SinQ[u[[1]]],
      (1-x^2)^(u[[2]]/2),
    If[CosQ[u[[1]]],
      x^u[[2]],
    If[TanQ[u[[1]]],
      (1-x^2)^(u[[2]]/2)/x^u[[2]],
    If[CotQ[u[[1]]],
      x^u[[2]]/(1-x^2)^(u[[2]]/2),
    If[SecQ[u[[1]]],
      1/x^u[[2]],
    1/(1-x^2)^(u[[2]]/2)]]]]],
  Map[Function[SubstForCos[#,v,x]],u]]]]]]]


(* u is a function of the form f(Tan[v],Cot[v]).  SubstForTan[u,v,x] returns f(x,1/x). *)
SubstForTan[u_,v_,x_] :=
  If[AtomQ[u],
    u,
  If[TanQ[u] && u[[1]]===v,
    x,
  If[CotQ[u] && u[[1]]===v,
    1/x,
  If[PowerQ[u] && EvenQ[u[[2]]] && TrigQ[u[[1]]] && u[[1,1]]===v,
    If[SinQ[u[[1]]],
      x^u[[2]]/(1+x^2)^(u[[2]]/2),
    If[CosQ[u[[1]]],
      1/(1+x^2)^(u[[2]]/2),
    If[TanQ[u[[1]]],
      x^u[[2]],
    If[CotQ[u[[1]]],
      1/x^u[[2]],
    If[SecQ[u[[1]]],
      (1+x^2)^(u[[2]]/2),
    (1+x^2)^(u[[2]]/2)/x^u[[2]]]]]]],
  Map[Function[SubstForTan[#,v,x]],u]]]]]


(* ::Subsubsection::Closed:: *)
(*Substitution For Hyperbolic Functions*)


(* u is a function of the form f(Sinh[v],Csch[v],Cosh[v/2]*Sinh[v/2]).
	SubstForSinh[u,v,x] returns f(x,1/x,x/2). *)
SubstForSinh[u_,v_,x_] :=
  If[AtomQ[u],
    u,
  If[SinhQ[u] && u[[1]]===v,
    x,
  If[CschQ[u] && u[[1]]===v,
    1/x,
  If[CoshQ[u] && u[[1]]===2*v,
    1+2*x^2,
  If[SechQ[u] && u[[1]]===2*v,
    1/(1+2*x^2),
  If[PowerQ[u] && EvenQ[u[[2]]] && HyperbolicQ[u[[1]]] && u[[1,1]]===v,
    If[SinhQ[u[[1]]],
      x^u[[2]],
    If[CoshQ[u[[1]]],
      (1+x^2)^(u[[2]]/2),
    If[TanhQ[u[[1]]],
      x^u[[2]]/(1+x^2)^(u[[2]]/2),
    If[CothQ[u[[1]]],
      (1+x^2)^(u[[2]]/2)/x^u[[2]],
    If[SechQ[u[[1]]],
      1/(1+x^2)^(u[[2]]/2),
    1/x^u[[2]]]]]]],
  If[ProductQ[u] && CoshQ[u[[1]]] && SinhQ[u[[2]]] && u[[1,1]]===u[[2,1]]===v/2,
    x/2*SubstForSinh[Drop[u,2],v,x],
  Map[Function[SubstForSinh[#,v,x]],u]]]]]]]]


(* u is a function of the form f(Cosh[v],Sech[v]).  SubstForCosh[u,v,x] returns f(x,1/x). *)
SubstForCosh[u_,v_,x_] :=
  If[AtomQ[u],
    u,
  If[CoshQ[u] && u[[1]]===v,
    x,
  If[SechQ[u] && u[[1]]===v,
    1/x,
  If[CoshQ[u] && u[[1]]===2*v,
    -1+2*x^2,
  If[SechQ[u] && u[[1]]===2*v,
    -1/(1-2*x^2),
  If[PowerQ[u] && EvenQ[u[[2]]] && HyperbolicQ[u[[1]]] && u[[1,1]]===v,
    If[SinhQ[u[[1]]],
      (-1+x^2)^(u[[2]]/2),
    If[CoshQ[u[[1]]],
      x^u[[2]],
    If[TanhQ[u[[1]]],
      (-1+x^2)^(u[[2]]/2)/x^u[[2]],
    If[CothQ[u[[1]]],
      x^u[[2]]/(-1+x^2)^(u[[2]]/2),
    If[SechQ[u[[1]]],
      1/x^u[[2]],
    1/(-1+x^2)^(u[[2]]/2)]]]]],
  Map[Function[SubstForCosh[#,v,x]],u]]]]]]]


(* u is a function of the form f(Tanh[v],Coth[v]).  SubstForTanh[u,v,x] returns f(x,1/x). *)
SubstForTanh[u_,v_,x_] :=
  If[AtomQ[u],
    u,
  If[TanhQ[u] && u[[1]]===v,
    x,
  If[CothQ[u] && u[[1]]===v,
    1/x,
  If[PowerQ[u] && EvenQ[u[[2]]] && HyperbolicQ[u[[1]]] && u[[1,1]]===v,
    If[SinhQ[u[[1]]],
      x^u[[2]]/(1-x^2)^(u[[2]]/2),
    If[CoshQ[u[[1]]],
      1/(1-x^2)^(u[[2]]/2),
    If[TanhQ[u[[1]]],
      x^u[[2]],
    If[CothQ[u[[1]]],
      1/x^u[[2]],
    If[SechQ[u[[1]]],
      (1-x^2)^(u[[2]]/2),
    (1-x^2)^(u[[2]]/2)/x^u[[2]]]]]]],
  Map[Function[SubstForTanh[#,v,x]],u]]]]]


(* ::Subsection::Closed:: *)
(*Substitute for trig functions*)


(* u(x) is equivalent to an expression of the form f(Sin[x],Cos[x],Tan[x],Cot[x],Sec[x],Csc[x])
	where f is independent of x. SubstForTrig[u,x] returns the expression
	f(2*x/(1+x^2),(1-x^2)/(1+x^2),2*x/(1-x^2),(1-x^2)/(2*x),(1+x^2)/(1-x^2),(1+x^2)/(2*x)). *)
SubstForTrig[u_,x_Symbol] :=
  If[AtomQ[u],
    u,
  If[TrigQ[u] && IntegerQ[u[[1]]/x],
    If[CotQ[u],
      1/SubstForTrig[Tan[u[[1]]],x],
    If[SecQ[u],
      1/SubstForTrig[Cos[u[[1]]],x],
    If[CscQ[u],
      1/SubstForTrig[Sin[u[[1]]],x],
    Module[{n=u[[1]]/x},
    If[n<0,
      If[SinQ[u],
        -SubstForTrig[Sin[-u[[1]]],x],
      If[CosQ[u],
        SubstForTrig[Cos[-u[[1]]],x],
      -SubstForTrig[Tan[-u[[1]]],x]]],
    If[n==1,
      If[SinQ[u],
        2*x/(1+x^2),
      If[CosQ[u],
        (1-x^2)/(1+x^2),
      2*x/(1-x^2)]],
    If[n==2,
      If[SinQ[u],
        4*x*(1-x^2)/(1+x^2)^2,
      If[CosQ[u],
        (1-6*x^2+x^4)/(1+x^2)^2,     (* Equivalent alternative: 1-8*x^2/(1+x^2)^2 *)
      4*x*(1-x^2)/(1-6*x^2+x^4)]],
    If[SinQ[u],
      2*(1-x^2)/(1+x^2)*SubstForTrig[Sin[(n-1)*x],x]-SubstForTrig[Sin[(n-2)*x],x],
    If[CosQ[u],
      2*(1-x^2)/(1+x^2)*SubstForTrig[Cos[(n-1)*x],x]-SubstForTrig[Cos[(n-2)*x],x],
    SubstForTrig[Sin[u[[1]]],x]/SubstForTrig[Cos[u[[1]]],x]]]]]]]]]],
  Map[Function[SubstForTrig[#,x]],u]]]


(* ::Subsection::Closed:: *)
(*Substitute for hyperbolic functions*)


(* u(x) is equivalent to an expression of the form f(Sin[x],Cos[x],Tan[x],Cot[x],Sec[x],Csc[x])
	where f is independent of x. SubstForHyperbolic[u,x] returns the expression
	f(2*x/(1-x^2),(1+x^2)/(1-x^2),2*x/(1+x^2),(1+x^2)/(2*x),(1-x^2)/(1+x^2),(1-x^2)/(2*x)). *)
SubstForHyperbolic[u_,x_Symbol] :=
  If[AtomQ[u],
    u,
  If[HyperbolicQ[u] && IntegerQ[u[[1]]/x],
    If[CothQ[u],
      1/SubstForHyperbolic[Tanh[u[[1]]],x],
    If[SechQ[u],
      1/SubstForHyperbolic[Cosh[u[[1]]],x],
    If[CschQ[u],
      1/SubstForHyperbolic[Sinh[u[[1]]],x],
    Module[{n=u[[1]]/x},
    If[n<0,
      If[SinhQ[u],
        -SubstForHyperbolic[Sinh[-u[[1]]],x],
      If[CoshQ[u],
        SubstForHyperbolic[Cosh[-u[[1]]],x],
      -SubstForHyperbolic[Tanh[-u[[1]]],x]]],
    If[n==1,
      If[SinhQ[u],
        2*x/(1-x^2),
      If[CoshQ[u],
        (1+x^2)/(1-x^2),
      2*x/(1+x^2)]],
    If[n==2,
      If[SinhQ[u],
        4*x*(1+x^2)/(1-x^2)^2,
      If[CoshQ[u],
        (1+6*x^2+x^4)/(1-x^2)^2,
      4*x*(1+x^2)/(1+6*x^2+x^4)]],
    If[SinhQ[u],
      2*(1+x^2)/(1-x^2)*SubstForHyperbolic[Sinh[(n-1)*x],x]-SubstForHyperbolic[Sinh[(n-2)*x],x],
    If[CoshQ[u],
      2*(1+x^2)/(1-x^2)*SubstForHyperbolic[Cosh[(n-1)*x],x]-SubstForHyperbolic[Cosh[(n-2)*x],x],
    SubstForHyperbolic[Sinh[u[[1]]],x]/SubstForHyperbolic[Cosh[u[[1]]],x]]]]]]]]]],
  Map[Function[SubstForHyperbolic[#,x]],u]]]


(* ::Subsection::Closed:: *)
(*Substitute for fractional powers*)


(* RootSubst[u,v,n,w,x] returns u(x) with fractional powers of the form v^m replaced by x^(n*m)
	and x replaced by w. *)
RootSubst[u_,v_,n_,w_,x_Symbol] :=
  If[AtomQ[u] || IndependentQ[u,x],
    If[u===x, w, u],
  If[FractionalPowerQ[u] && u[[1]]===v,
    x^(n*u[[2]]),
  Map[Function[RootSubst[#,v,n,w,x]],u]]]


(* ::Section::Closed:: *)
(*Derivative Divides Function*)


(* If u is easy to differentiate wrt x and the derivative divides v wrt x, returns the quotient;
   else it returns False. *)
DerivativeDivides[u_,v_,x_Symbol] :=
  If[PolynomialQ[u,x],
    If[PolynomialQ[v,x] && Exponent[v,x]==Exponent[u,x]-1,
      Module[{w=Block[{ShowSteps=False}, Simplify[v/D[u,x]]]},
      If[FreeQ[w,x],
        w,
      False]],
    False],      
  If[EasyDQ[u,x],
    Module[{w=Block[{ShowSteps=False}, Simplify[v/D[u,x]]]},
    If[FreeQ[w,x],
      w,
    False]],
  False]]


(* If u is easy to differentiate wrt x, returns True; else it returns False. *)
EasyDQ[u_,x_Symbol] :=
  If[AtomQ[u] || IndependentQ[u,x] || Length[u]==0,
    True,
  If[CalculusQ[u],
    False,
  If[Length[u]==1,
    EasyDQ[u[[1]],x],
  If[ProductQ[u],
    If[FreeQ[First[u],x],
      EasyDQ[Rest[u],x],
    If[FreeQ[Rest[u],x],
      EasyDQ[First[u],x],
    False]],
  If[SumQ[u],
    EasyDQ[First[u],x] && EasyDQ[Rest[u],x],
  If[Length[u]==2,
    If[FreeQ[u[[1]],x],
      EasyDQ[u[[2]],x],
    If[FreeQ[u[[2]],x],
      EasyDQ[u[[1]],x],
    False]],
  False]]]]]]


(* ::Section::Closed:: *)
(*Nth Root Function*)


Rt[u_^m_,n_Integer] :=
  1/Rt[u^-m,n] /;
RationalQ[m] && m<0

Rt[u_^m_,n_Integer] :=
  Rt[u^(1/Denominator[m]),n/GCD[Numerator[m],n]]^(Numerator[m]/GCD[Numerator[m],n]) /;
RationalQ[m] && Numerator[m]>1

Rt[u_*v_,n_Integer] :=
  Rt[u,n]*Rt[v,n] /;
OddQ[n] || Not[NegativeOrZeroQ[u]] && Not[NegativeOrZeroQ[v]]

Rt[u_,n_Integer] :=
  Rt[Simplify[u],n] /;
Simplify[u]=!=u

Rt[u_,n_Integer] :=
  u^(1/n)


(* ::Section::Closed:: *)
(*Vector Function*)


SetAttributes[Vector,HoldFirst]

Vector[u_,x_Symbol,a_,b_] :=
  Table[u,{x,a,b}]

Vector[u_,x_Symbol,a_,b_,c_] :=
  Table[u,{x,a,b,c}]
